<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Util\UtilMethods;
use App\Models\AppHistoryThirdPartyOutdoor;
use App\Models\KycThirdParty;
use App\Models\ThirdPartyOffchargeAndExpunction;
use App\Models\ThirdPartyOutdoor;
use App\Models\ThirdPartyOutdoorSignDetails;
use App\Models\ThirdPartyOutdoorTransfer;
use App\Models\ThirdPartyUpgradeAndDowngrade;
use App\Models\User;
use App\Models\UserAdmin;
use App\Models\ThirdPartyReceipts;
use App\Models\OutdoorSignagePricing;
use App\Models\OutdoorSignagePricingApplication;
use App\Models\EmailContent;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use PHPMailer\PHPMailer\Exception;
use PHPMailer\PHPMailer\PHPMailer;

class AdminThirdPartySignageController extends Controller {

    private $successStatusCode = 200;
    private $createStatusCode = 201;
    private $errorStatusCode = 422;

    public function getAllOutdoorSignage(Request $request) {

        $keyword = $request->input('search');
        $getAllOutdoorSignage = ThirdPartyOutdoor::with(['user', 'thirdPartyOutdoorSignDetails', 'thirdPartyOffChargeAndExpunction',
            'thirdPartyUpgradeAndDowngrade', 'thirdPartyOutdoorTransfer']);

        if($keyword) {
            $getAllOutdoorSignage->where(function ($query) use ($keyword) {
                $query->where('purpose_application', '%' . $keyword . '%')
                    ->where('address_proposed_site', '%' . $keyword .  '%')
                    ->where('lga', '%' . $keyword . '%')
                    ->where('property_owner_name', '%' . $keyword . '%');
            });

            $getAllOutdoorSignage->orWhereHas('user', function ($query) use ($keyword) {
                $query->where('first_name', '%' . $keyword . '%')
                    ->where('last_name', '%' . $keyword . '%')
                    ->where('email', '%' . $keyword . '%')
                    ->where('phone_number', '%' . $keyword . '%')
                    ->where('id', '%' . $keyword . '%');
            });

            $getAllOutdoorSignage->orWhereHas('thirdPartyOutdoorSignDetails', function ($query) use ($keyword) {
                $query->where('signage_format', '%' . $keyword . '%')
                    ->where('sign_interaction', '%' . $keyword . '%')
                    ->where('sign_illumination', '%' . $keyword . '%')
                    ->where('number_faces', '%' . $keyword . '%');
            });

        }
        
        if ($request->has('lga') && $request->input('lga') !="") {
            $search = $request->input('lga');
            $getAllOutdoorSignage->where('lga', $search);
        }
                
                
        if ($request->has('start_date') && $request->input('start_date') !="") {
            $startDate = Carbon::parse($request->input('start_date'));
            $endDate = Carbon::parse($request->input('end_date'));
            $getAllOutdoorSignage->whereBetween('created_at', [$startDate, $endDate]);
        }
        $this->annualOutdoorBills();
        $getAllItems = $getAllOutdoorSignage->orderBy('created_at', 'desc')->paginate(20);
        return response()->json(['success' => true, 'data' => $getAllItems], $this->successStatusCode);
    }

    public function getOneOutdoorSignage($outdoorId) {
        $getOneOutdoor = ThirdPartyOutdoor::where('id', $outdoorId)
            ->with(['user', 'thirdPartyOutdoorSignDetails'])
            ->with(['thirdPartyUpgradeAndDowngrade' => function($query) {
                $query->latest()->first();
            }])->with(['thirdPartyOutdoorTransfer' => function($query) {
                $query->latest()->first();
            }])->with(['thirdPartyOffChargeAndExpunction' => function($query) {
                $query->latest()->first();
            }])->first();
            
        
        if($getOneOutdoor['application_certify_status'] == "pending"){
            $calladmin = UserAdmin::select('first_name', 'last_name', 'department')->where('id', $getOneOutdoor->assigned_officer)->where('department', 'Registry')->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing relational manager admin";
            }
        }elseif($getOneOutdoor['hod_recommedation'] == "pending"){
            $callsub = UserAdmin::where('id', $getOneOutdoor->assigned_officer)->first();
            $calladmin = UserAdmin::select('first_name', 'last_name', 'department')->where('id', $callsub->line_manager)->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing hod admin";
            }
        }elseif($getOneOutdoor['md_first_approval'] == "pending"){
            $calladmin = "M.D";
        }elseif($getOneOutdoor['rm_second_recommendation'] == "pending"){
            $calladmin = UserAdmin::select('first_name', 'last_name', 'department')->where('id', $getOneOutdoor->assigned_officer)->where('department', 'Registry')->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing relational manager admin";
            }
        }elseif($getOneOutdoor['hod_second_recommendation'] == "pending"){
            $callsub = UserAdmin::where('id', $getOneOutdoor->assigned_officer)->first();
            $calladmin = UserAdmin::select('first_name', 'last_name', 'department')->where('id', $callsub->line_manager)->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing hod admin";
            }
        }elseif($getOneOutdoor['legal_recommendation'] == "pending"){
            $calladmin = UserAdmin::select('first_name', 'last_name')->where('admin_role_id', 9)->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing legal manager admin";
            }
        }elseif($getOneOutdoor['md_second_approval'] == "pending"){
            $calladmin = "M.D";
        }elseif($getOneOutdoor['bill_generated'] == 0){
            $calladmin = UserAdmin::select('first_name', 'last_name')->where('department', 'Billings')->where('admin_role_id', 2)->orWhere('admin_role_id', 3)->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing billings manager admin";
            }
        }elseif($getOneOutdoor['hod_bill_generate'] == "pending"){
            $calladmin = UserAdmin::select('first_name', 'last_name')->where('department', 'Billings')->where('admin_role_id', 5)->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing billings hod admin";
            }
        }elseif($getOneOutdoor['account_status'] == "pending"){
            $calladmin = UserAdmin::select('first_name', 'last_name')->where('admin_role_id', 8)->get()->toArray();
            if(empty($calladmin)){
                $calladmin = "missing finance manager admin";
            }
        }else{
            $calladmin = "";
        }
        

        if (is_null($getOneOutdoor)) {
            return response()->json(['success' => false, 'data' => 'There is no outdoor signage application with the outdoor id'], $this->errorStatusCode);
        }
        return response()->json(['success' => true, 'data' => $getOneOutdoor, 'admins' => $calladmin], $this->successStatusCode);
    }

    public function certifyOutdoorApplication($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3) {
                $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
                if($getOutdoorObj->application_certify_status == 'pending') {
                    if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                    $getApproval = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['application_certify_status' => 'approved', 'rm_comment1' => $bmcomment]);
                    $getData = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->get();
                    return response()->json(['success' => true, 'data' => $getData], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'application has been treated'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function  decertifyOutdoorApplication($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'required|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3) {
                $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
                if($getOutdoorObj->application_certify_status == 'pending') {
                    if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                    $getApproval = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['application_certify_status' => 'disapproved', 'rm_comment1' => $bmcomment]);
                    $getData = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->get();
                    
                    $getApplicant = User::where('id', $getData[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                    return response()->json(['success' => true, 'data' => $getData], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'application has been treated'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function  uploadSiteReportByRM(Request $request, $outdoorId) {

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3) {
                $validate = Validator::make($request->all(), [
                    'reportFile' => 'required|string',
                    // 'signageDesignation' => 'required|string',
                    // 'signageLength' => 'required|numeric',
                    // 'signageBreadth' => 'required|numeric',
                    // 'sides' => 'required|integer|min:1',
                    'installation_date' => 'required|date'
                ]);

                if ($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }

                $InputFile = $request->all();
                $getOudoorObj = ThirdPartyOutdoor::find($outdoorId);
                $getFileUrl = UtilMethods::convertBase64toPdf($InputFile['reportFile']);
                if($getOudoorObj->report_file == null) {
                    // $getDimension = ($InputFile['signageLength'] * $InputFile['signageBreadth']);
                    ThirdPartyOutdoor::where('id', $getOudoorObj->id)->update(['report_file' => $getFileUrl, 'hod_recommedation' => 'pending', 'inspection_date' => $InputFile['installation_date']]);
                    AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOudoorObj->id)->update(['who_performed_action' =>
                                            (string)Auth::user()->id, 'comments' => 'Site report uploaded']);
                    // ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOudoorObj->id)->update(['designation' =>
                    //                         $InputFile['signageDesignation'], 'length_sign' => $InputFile['signageLength'], 'breadth_sign' => $InputFile['signageBreadth'],
                    //                         'dimension_sign' => $getDimension, 'side' => $InputFile['sides'], 'number_faces' => $InputFile['sides']]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOudoorObj->id)
                                            ->with(['thirdPartyOutdoorSignDetails', 'appHistoryThirdPartyOutdoor'])->get();
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return  response()->json(['success' => false, 'data' => 'site report has already been uploaded for this signage'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function hodApprovalRecommendation($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 5) {
                $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
                if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                if($getOutdoorSignage->hod_recommedation == 'pending') {
                    ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['hod_recommedation' => 'approved', 'md_first_approval' => 'pending', 'hod_comment1' => $bmcomment]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'The signage has already been treated'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function hodDisapprovalRecommendation($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'required|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 5) {
                $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
                if($getOutdoorSignage->hod_recommedation == 'pending') {
                    if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                    ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['hod_recommedation' => 'disapproved', 'md_first_approval' => 'pending', 'hod_comment1' => $bmcomment]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                    $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'The signage has already been treated'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function mdApproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->admin_role_id == 6) {
            $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
            if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
            if($getOutdoorSignage->hod_recommedation !== 'pending' && $getOutdoorSignage->md_first_approval == 'pending') {
                ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['md_first_approval' =>'approved', 'md_comment' => $bmcomment]);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                
                $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_first_approve_content');
                    $subject = $emailsubject;
                    $body = $emailcontent;
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                    
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'Either hod recommendation is pending or signage has been treated']
                                    , $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
        }
    }

    public function mdDisapproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'required|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->admin_role_id == 6) {
            $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
            if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
            if($getOutdoorSignage->hod_recommedation !== 'pending' && $getOutdoorSignage->md_first_approval == 'pending') {
                ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['md_first_approval' =>'disapproved', 'md_comment' => $bmcomment]);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'Either hod recommendation is pending or signage has been treated']
                    , $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
        }
    }

    //Second Stage of the outdoor Signage

    public function rmSecondStageDocReviewApproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3) {
                $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
                if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                if ($getOutdoorObj->rm_second_recommendation == 'pending' && $getOutdoorObj->md_first_approval == 'approved' ) {
                    ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['rm_second_recommendation' => 'approved',
                                                                'hod_second_recommendation' => 'pending', 'rm_comment2' => $bmcomment]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->get();
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'The signage has been treated']
                        , $this->errorStatusCode);
                    }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function rmSecondStageDocReviewDisapproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'required|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3) {
                if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
                if ($getOutdoorObj->rm_second_recommendation == 'pending' && $getOutdoorObj->md_first_approval == 'approved' ) {
                    ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['rm_second_recommendation' => 'disapproved',
                        'hod_second_recommendation' => 'pending', 'rm_comment2' => $bmcomment]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->get();
                    $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'The signage has been treated']
                        , $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function hodSecondStageRecommendationApproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 5) {
                $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
                if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                if($getOutdoorSignage->hod_second_recommendation == 'pending') {
                    ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['hod_second_recommendation' => 'approved',
                                                                    'legal_recommendation' => 'pending', 'hod_comment2' => $bmcomment]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'The signage has already been treated'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function hodSecondStageRecommendationDisapproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'required|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if(Auth::user()->department == 'Registry') {
            if(Auth::user()->admin_role_id == 5) {
                $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
                if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
                if($getOutdoorSignage->hod_second_recommendation == 'pending') {
                    ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['hod_second_recommendation' => 'disapproved',
                        'legal_recommendation' => 'pending', 'hod_comment2' => $bmcomment]);
                    $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                    $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                    return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'The signage has already been treated'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'can not access the resource due to incorrect department']
                , $this->errorStatusCode);
        }
    }

    public function legalRecommendationApproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if (Auth::user()->admin_role_id == 9) {
            $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
            if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
            if($getOutdoorSignage->legal_recommendation == 'pending') {
                ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['legal_recommendation' => 'approved', 'md_second_approval' => 'pending', 'legal_comment' => $bmcomment]);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'The signage has already been treated'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you can not access this resource.'], $this->errorStatusCode);
        }
    }

    public function legalRecommendationDisapproval($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'required|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        if (Auth::user()->admin_role_id == 9) {
            $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
            if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
            if($getOutdoorSignage->legal_recommendation == 'pending') {
                ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['legal_recommendation' => 'disapproved', 'md_second_approval' => 'pending', 'legal_comment' => $bmcomment]);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'The signage has already been treated'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you can not access this resource.'], $this->errorStatusCode);
        }
    }

    public function mdSecondApproval($outdoorId) {

        if(Auth::user()->admin_role_id == 6) {
            $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
            if($getOutdoorSignage->legal_recommendation !== 'pending' && $getOutdoorSignage->md_second_approval == 'pending') {
                ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['md_second_approval' => 'approved']);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            }  else {
                return response()->json(['success' => false, 'data' => 'Either legal recommendation is pending or signage has been treated']
                    , $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
        }
    }

    public function mdSecondDisApproval($outdoorId) {

        if(Auth::user()->admin_role_id == 6) {
            $getOutdoorSignage = ThirdPartyOutdoor::find($outdoorId);
            if($getOutdoorSignage->legal_recommendation !== 'pending' && $getOutdoorSignage->md_second_approval == 'pending') {
                ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->update(['md_second_approval' => 'disapproved']);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorSignage->id)->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            }  else {
                return response()->json(['success' => false, 'data' => 'Either legal recommendation is pending or signage has been treated']
                    , $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'do not have the required role'], $this->errorStatusCode);
        }
    }
    
    public function getOutdoorPricing (){
        
        $getPrices = OutdoorSignagePricing::all();
        return response()->json(['success' => true, 'data' => $getPrices], 200);
    }
    
    public function getOutdoorPricingDistinct(){
        
        $getPrices = OutdoorSignagePricing::distinct()->pluck('designation');
        return response()->json(['success' => true, 'data' => $getPrices], 200);
    }
    
    public function deleteOutdoorPricing($id){
        
        $getPrices = OutdoorSignagePricing::find($id)->delete();
        return response()->json(['success' => true, 'message' => 'Pricing deleted successfully',], 200);
    }
    
    public function getOutdoorPricingApplication (){
        
        $getPrices = OutdoorSignagePricingApplication::all();
        return response()->json(['success' => true, 'data' => $getPrices], 200);
    }
    
    public function updateOutdoorPricing(Request $request, $id) {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'designation' => 'sometimes|string|max:255',
            'street_category' => 'sometimes|string|max:255',
            'dimension_min' => 'nullable|numeric',
            'dimension_max' => 'nullable|numeric',
            'base_rate' => 'sometimes|numeric|min:0',
            'multiplier' => 'sometimes|numeric|min:0',
            'calculation_type' => 'sometimes|in:daily,weekly'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 400);
        }

        // Find the pricing record
        $pricing = OutdoorSignagePricing::find($id);
        
        if (!$pricing) {
            return response()->json([
                'success' => false,
                'message' => 'Pricing record not found'
            ], 404);
        }

        // Update the record
        $pricing->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Pricing updated successfully',
            'data' => $pricing
        ]);
    }
    
    public function createOutdoorPricing(Request $request) {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'designation' => 'sometimes|string|max:255',
            'street_category' => 'sometimes|string|max:255',
            'dimension_min' => 'nullable|numeric',
            'dimension_max' => 'nullable|numeric',
            'base_rate' => 'sometimes|numeric|min:0',
            'multiplier' => 'sometimes|numeric|min:0',
            'calculation_type' => 'sometimes|in:daily,weekly'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 400);
        }

        // Update the record
       $pricing = OutdoorSignagePricing::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Pricing updated successfully',
            'data' => $pricing
        ]);
    }
    
    public function updateOutdoorPricingApplication(Request $request, $id) {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'led_price' => 'sometimes|numeric|min:0',
            'normal_price' => 'sometimes|numeric|min:0',
            'inspection_fee' => 'sometimes|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 400);
        }

        // Find the pricing record
        $pricing = OutdoorSignagePricingApplication::find($id);
        
        if (!$pricing) {
            return response()->json([
                'success' => false,
                'message' => 'Pricing record not found'
            ], 404);
        }

        // Update the record
        $pricing->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Pricing updated successfully',
            'data' => $pricing
        ]);
    }
    
    public function billGeneration(Request $request, $outdoorId) {
        
        if((Auth::user()->department == 'Billings') && (Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3 || Auth::user()->admin_role_id == 5)) {
            $validate = Validator::make($request->all(), [
                'streetClassification' => 'required|string',
                'ads_price' => 'nullable|numeric|between:0,99999999.99',
                'price_rate' => 'nullable|numeric|between:0,99999999.99',
                'ads_price_pyears' => 'nullable|numeric|between:0,99999999.99',
            ]);
    
            if($validate->fails()) {
                return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
            }
    
            $billingInput = $request->all();
            $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
    
            if($getOutdoorObj->md_second_approval == 'approved') {
                // Update street classification
                ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)
                    ->update(['street_category' => $billingInput['streetClassification']]);
    
                $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)->first();
                
                // Calculate billing period
                $getApprovedDate = Carbon::parse($getOutdoorObj->updated_at);
                // Test case(One year)
                // $getApprovedDate = Carbon::createFromDate(Carbon::now()->year, 1, 1)->subMonth();
                $getPaymentStartDate = $getApprovedDate->addMonth();
                
                $lastDayofYear = Carbon::createFromDate(Carbon::now()->year, 12, 31);
                $lastDayofYearLastYear = Carbon::createFromDate(Carbon::now()->year - 1, 12, 31);
                $daysRemaining = $getPaymentStartDate->diffInDays($lastDayofYear) + 1;
                $weeksRemaining = min($getPaymentStartDate->diffInWeeks($lastDayofYear) + 1, 52);
                
                
                
                if($getPaymentStartDate->year == Carbon::now()->year) {
                    $daysRemainingPYears = 0;
                    $weeksRemainingPYears = 0;
                } else {
                    $daysRemainingPYears = $getPaymentStartDate->diffInDays($lastDayofYearLastYear);
                    $weeksRemainingPYears = $getPaymentStartDate->diffInWeeks($lastDayofYearLastYear);
                }
                
                $currentYear = Carbon::now()->year;
                $numberDaysInYear = Carbon::createFromDate($currentYear, 12, 31)->dayOfYear;
    
                // Get pricing from database
                $pricing = OutdoorSignagePricing::where('designation', $getOutdoorDetails->designation)
                    ->where('street_category', $getOutdoorDetails->street_category)
                    ->where(function($query) use ($getOutdoorDetails) {
                        $query->whereNull('dimension_max')
                            ->orWhere('dimension_max', '>=', $getOutdoorDetails->dimension_sign);
                    })
                    ->where(function($query) use ($getOutdoorDetails) {
                        $query->whereNull('dimension_min')
                            ->orWhere('dimension_min', '<=', $getOutdoorDetails->dimension_sign);
                    })
                    ->first();
    
                if (!$pricing) {
                    return response()->json([
                        'success' => false, 
                        'data' => 'Pricing not configured for this signage type: ' . $getOutdoorDetails->designation
                    ], $this->errorStatusCode);
                }
    
                // Calculate price based on dimension
                $dimension = $getOutdoorDetails->dimension_sign;
                $baseRate = $pricing->base_rate;
                
                // Apply dimension multiplier if needed
                if ($pricing->dimension_max && $dimension > $pricing->dimension_max) {
                    $baseRate = $baseRate * ($dimension / $pricing->dimension_max) * $pricing->multiplier;
                }
                
    
                // Calculate daily or weekly rate
                if ($pricing->calculation_type === 'daily') {
                    $amountPerDay = $baseRate / $numberDaysInYear;
                    $adsPrice = $amountPerDay * $daysRemaining;
                    $adsPricePYears = $amountPerDay * $daysRemainingPYears;
                } else {
                    // Weekly calculation
                    $adsPrice = $baseRate * $weeksRemaining;
                    $adsPricePYears = $baseRate * $weeksRemainingPYears;
                }
    
                // Apply side multiplier
                $adsPrice = $adsPrice * $getOutdoorDetails->side;
                $adsPricePYears = $adsPricePYears * $getOutdoorDetails->side;
                
                // Round values
                $adsPrice = round($adsPrice, 2);
                $adsPricePYears = round($adsPricePYears, 2);
                
                // Calculate legal fees
                $legalFee = (1 / 100) * $adsPrice;
                $legalFee = round($legalFee, 2);
                $legalFeePYears = (1 / 100) * $adsPricePYears;
                $legalFeePYears = round($legalFeePYears, 2);
                
                $getRate = $baseRate;
    
                // Handle different admin roles
                if(Auth::user()->admin_role_id == 5) {
                    // HOD can override prices
                    if (isset($billingInput['ads_price'])) {
                        $legalFee = (1 / 100) * $billingInput['ads_price'];
                        $adsPrice = $billingInput['ads_price'];
                        $getRate = $billingInput['price_rate'];
                        $adsPricePYears = $billingInput['ads_price_pyears'];
                        $legalFeePYears = (1 / 100) * $billingInput['ads_price_pyears'];
                    }
    
                    // Update application history
                    AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOutdoorObj->id)
                        ->update([
                            'ads_price' => $adsPrice,
                            'legal_fee' => $legalFee, 
                            'price_rate' => $getRate
                        ]);
    
                    // Update sign details
                    ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)
                        ->update([
                            'ads_price' => $adsPrice,
                            'legal_fee' => $legalFee, 
                            'price_rate' => $getRate,
                            'ads_price_pyears' => $adsPricePYears, 
                            'legal_fee_pyears' => $legalFeePYears,
                            'ads_price_current_year' => $adsPrice - $adsPricePYears,
                            'legal_fee_current_year' => $legalFee - $legalFeePYears
                        ]);
    
                    // Get complete application details
                    $getThirdPartyDetails = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)
                        ->with(['user', 'thirdPartyOutdoorSignDetails'])
                        ->with(['appHistoryThirdPartyOutdoor' => function($query) {
                            $query->latest()->first();
                        }])->first();
    
                    $userDetails = KycThirdParty::where('user_id', $getThirdPartyDetails->user_id)
                        ->with(['user'])
                        ->first();
    
                    $totalFee = bcadd($adsPrice, $legalFee, 2);
                    $amountInWord = UtilMethods::getAmountInWords($totalFee);
    
                    $getDate = Carbon::parse($getThirdPartyDetails->updated_at);
                    $getYear = $getDate->year;
                    if($getApprovedDate->year < $getYear) {
                        $getApprovedDate = $getDate->startOfYear();
                    }
    
                    // Prepare data for invoice
                    $data = [
                        'custId' => $userDetails->user->id,
                        'email' => $userDetails->user->email,
                        'userType' => $userDetails->user->user_type,
                        'company' => $userDetails->company_name,
                        'officeAddress' => $userDetails->office_address,
                        'billYear' => $getYear,
                        'signageLocation' => $getThirdPartyDetails->address_proposed_site,
                        'appId' => $getThirdPartyDetails->id,
                        'billStartDate' => $getApprovedDate->format('m-d-y'),
                        'billEndDate' => $getDate->endOfYear()->format('m-d-y'),
                        'dimension' => $getThirdPartyDetails->thirdPartyOutdoorSignDetails->dimension_sign,
                        'signage' => $getThirdPartyDetails->thirdPartyOutdoorSignDetails->designation,
                        'zone' => $getThirdPartyDetails->thirdPartyOutdoorSignDetails->street_category,
                        'side' => $getThirdPartyDetails->thirdPartyOutdoorSignDetails->side,
                        'rate' => $getThirdPartyDetails->appHistoryThirdPartyOutdoor[0]->price_rate,
                        'adsFee' => $getThirdPartyDetails->appHistoryThirdPartyOutdoor[0]->ads_price,
                        'legalFee' => $getThirdPartyDetails->appHistoryThirdPartyOutdoor[0]->legal_fee,
                        'totalFee' => $totalFee,
                        'amountInWord' => $amountInWord
                    ];
    
                    // Generate PDF invoice
                    $pdf = Pdf::loadView('BillInvoice', compact('data'))->setPaper('A3');
                    $fileObj = $pdf->download('BillInvoice.pdf');
    
                    $getUserObj = User::find($userDetails->user->id);
                    
                    $emailcontent = EmailContent::getContent('outdoor_bill_content');
    
                    // Send email with invoice
                    $body = $emailcontent;
                    $templateData = [
                        'body' => $body,
                        'name' => $getUserObj->company_name
                    ];
                    $firstBillYear = Carbon::now()->year;
                    $templateContent = view('hodeoitemplate', $templateData)->render();
                    UtilMethods::outdoorSignageInvoiceBill($getUserObj->email, $templateContent, $fileObj);
                    
                    // Save invoice to storage
                    $pdfContent = $pdf->output();
                    $filename = 'BillInvoice_' . now()->format('YmdHis') . '.pdf';
                    $relativePath = 'invoices/' . $filename;
                    
                    if (!File::exists(public_path('invoices'))) {
                        File::makeDirectory(public_path('invoices'), 0755, true);
                    }
                    
                    file_put_contents(public_path($relativePath), $pdfContent);
                    $fileUrl = config('app.url') . '/public/' . $relativePath;
                    
                    // Create receipt record
                    $adnewrec = new ThirdPartyReceipts();
                    $adnewrec->link = $fileUrl;
                    $adnewrec->type = "Invoice";
                    $getOutdoorObj->thirdpartyreceipts()->save($adnewrec);
                    
                    // Update application status
                    ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update([
                        'hod_bill_generate' => 'approved', 
                        'permit_year' => $firstBillYear
                    ]);
                    
                    if($adsPricePYears > 0) {
                        ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['account_status' => 'pending']);
                    } else {
                        ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['account_status' => 'approved']);
                    }
                    
                } elseif(in_array(Auth::user()->admin_role_id, [2, 3])) {
                    // Billing staff can't override prices
                    if($billingInput['streetClassification'] == "Not Applicable") {
                        $legalFee = (1 / 100) * $billingInput['ads_price'];
                        $adsPrice = $billingInput['ads_price'];
                        $getRate = $billingInput['price_rate'];
                        $adsPricePYears = $billingInput['ads_price_pyears'];
                        $legalFeePYears = (1 / 100) * $billingInput['ads_price_pyears'];
                    }
                    
                    // Update records
                    AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOutdoorObj->id)
                        ->update([
                            'ads_price' => $adsPrice,
                            'legal_fee' => $legalFee, 
                            'price_rate' => $getRate
                        ]);
    
                    ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)
                        ->update([
                            'ads_price' => $adsPrice,
                            'legal_fee' => $legalFee, 
                            'price_rate' => $getRate,
                            'ads_price_pyears' => $adsPricePYears, 
                            'legal_fee_pyears' => $legalFeePYears,
                            'ads_price_current_year' => $adsPrice - $adsPricePYears,
                            'legal_fee_current_year' => $legalFee - $legalFeePYears
                        ]);
    
                    ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update([
                        'bill_generated' => true, 
                        'hod_bill_generate' => 'pending'
                    ]);
                }
    
                return response()->json([
                    'success' => true, 
                    'data' => 'Invoice sent', 
                    'signage bill' => $adsPrice, 
                    'legal fee' => $legalFee
                ], $this->successStatusCode);
                
            } else {
                return response()->json([
                    'success' => false, 
                    'data' => 'The outdoor signage has not been approved'
                ], $this->errorStatusCode);
            }
        } else {
            return response()->json([
                'success' => false, 
                'data' => 'Only billings role can perform this action'
            ], $this->errorStatusCode);
        }
    }
    
    public function accountApprovedRequest($outdoorId, Request $request) {
        $validate = Validator::make($request->all(),  [
            'comments' => 'nullable|string|max:255',
            'reports_from_branchmanger' => 'required|string|max:255',
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], 422);
        }

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);

        if($getOutdoorObj->account_status == 'approved') {
                return response()->json(['success' => false, 'data' => 'The request has been treated'], $this->errorStatusCode);
        } else {
            if(isset($request['reports_from_branchmanger'])){
                        $bmcomment = $request['reports_from_branchmanger'];
                    }else{
                        $bmcomment = "";
                    }
                    if(isset($request['comments'])){
                        $ucomment = $request['comments'];
                    }else{
                        $ucomment = "";
                    }
            $getOutdoorObj->account_status = 'approved';
            $getOutdoorObj->account_comment = $bmcomment;
            $getOutdoorObj->save();
            return response()->json(['success' => true, 'data' => 'Completed'], $this->successStatusCode);
        }
    }

    public function complianceApprovedRequest(Request $request, $outdoorId, $requestId) {
        $validate = Validator::make($request->all(), [
            'complianceReport' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $assignedOfficer = $getOutdoorObj->assigned_officer;

        if((string)Auth::user()->id == (string)$assignedOfficer) {
                $requestObj = ThirdPartyOffchargeAndExpunction::find($requestId);
              if($requestObj->complaince_confirm_status == 'pending') {
                  $siteSubmission = UtilMethods::convertBase64toPdf($request['complianceReport']);
                  ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->update(['complaince_confirm_status'=> 'approved', 'hod_recommedation'=> 'pending', 'compliance_report' => $siteSubmission]);
                  $getResponse = ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->get();
                  return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
              }
                return response()->json(['success' => false, 'data' => 'The request has been treated'], $this->errorStatusCode);
        } else {
            return response()->json(['success' => false, 'data' => 'Not the RM assigned to the signage'], $this->errorStatusCode);
        }
    }
    
    public function rmApprovedRequest(Request $request, $outdoorId, $requestId) {
            $validate = Validator::make($request->all(), [
            'siteSubmission' => 'nullable|string',
            'comment' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $assignedOfficer = $getOutdoorObj->assigned_officer;

        if((string)Auth::user()->id == (string)$assignedOfficer) {
                $requestObj = ThirdPartyOffchargeAndExpunction::find($requestId);
              if($requestObj->rm_recommedation == 'pending') {
                  $siteSubmission = UtilMethods::convertBase64toPdf($request['siteSubmission']);
                  ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->update(['rm_recommedation'=> 'approved', 'rm_comment' => $request['comment'], 'site_submission' => $siteSubmission, 'complaince_confirm_status' => 'pending']);
                  $getResponse = ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->get();
                  return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
              }
                return response()->json(['success' => false, 'data' => 'The request has been treated'], $this->errorStatusCode);
        } else {
            return response()->json(['success' => false, 'data' => 'Not the RM assigned to the signage'], $this->errorStatusCode);
        }
    }
    
    public function hodApprovedRequest(Request $request, $outdoorId, $requestId) {
            $validate = Validator::make($request->all(), [
            'comment' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $assignedOfficer = $getOutdoorObj->assigned_officer;

        if((string)Auth::user()->id == (string)$assignedOfficer) {
                $requestObj = ThirdPartyOffchargeAndExpunction::find($requestId);
              if($requestObj->hod_recommedation == 'pending') {
                  ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->update(['hod_recommedation'=> 'approved', 'hod_comment' => $request['comment'], 'md_approval' => 'pending']);
                  $getResponse = ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->get();
                  return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
              }
                return response()->json(['success' => false, 'data' => 'The request has been treated'], $this->errorStatusCode);
        } else {
            return response()->json(['success' => false, 'data' => 'Not the RM assigned to the signage'], $this->errorStatusCode);
        }
    }
    
    public function hodDisapproveRequest(Request $request, $outdoorId, $requestId) {
            $validate = Validator::make($request->all(), [
            'comment' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $assignedOfficer = $getOutdoorObj->assigned_officer;

        if((string)Auth::user()->id == (string)$assignedOfficer) {
                $requestObj = ThirdPartyOffchargeAndExpunction::find($requestId);
              if($requestObj->hod_recommedation == null) {
                  ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->update(['hod_recommedation'=> 'disapproved', 'hod_comment' => $request['comment'], 'rm_recommedation' => 'disapproved', 'complaince_confirm_status' => 'disapproved']);
                  $getResponse = ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->get();
                  return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
              }
                return response()->json(['success' => false, 'data' => 'The request has been treated'], $this->errorStatusCode);
        } else {
            return response()->json(['success' => false, 'data' => 'Not the RM assigned to the signage'], $this->errorStatusCode);
        }
    }

    public function complianceDisapprovedRequest($outdoorId, $requestId) {

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $assignedOfficer = $getOutdoorObj->assigned_officer;

        if((string)Auth::user()->id == (string)$assignedOfficer) {
            $requestObj = ThirdPartyOffchargeAndExpunction::find($requestId);
            if($requestObj->complaince_confirm_status == null) {
                ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->update(['complaince_confirm_status' => 'disapproved']);
                $getResponse = ThirdPartyOffchargeAndExpunction::where('id', $requestObj->id)->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            }
            return response()->json(['success' => false, 'data' => 'The request has been treated'], $this->errorStatusCode);
        } else {
            return response()->json(['success' => false, 'data' => 'Not the RM assigned to the signage'], $this->errorStatusCode);
        }
    }

    public function outdoorSiteReportforRequest(Request $request, $outdoorId,$offchargeId) {
        $validate = Validator::make($request->all(), [
            'siteSubmission' => 'required|string',
            'complianceReport' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }

        $reports = $request->all();
        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $getOffchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);


        if(Auth::user()->id == $getOutdoorObj->assigned_officer) {
            if($getOffchargeObj->compliance_report == null && $getOffchargeObj->site_submission == null) {
                $siteSubmission = UtilMethods::convertBase64toPdf($reports['siteSubmission']);
                $complianceReport = UtilMethods::convertBase64toPdf($reports['complianceReport']);

                ThirdPartyOffchargeAndExpunction::where('third_party_outdoor_id', $getOutdoorObj->id)->update(['compliance_report' => $complianceReport,
                    'site_submission' => $siteSubmission]);
                $reportUploaded = ThirdPartyOffchargeAndExpunction::where('id', $offchargeId)->get();
                return response()->json(['success' => true, 'data' => $reportUploaded], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'report has been uploaded already'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Not the RM assigned to the signage'], $this->errorStatusCode);
        }
    }

    public function mdApprovalforRequest(Request $request, $outdoorId, $offchargeId) {
        $validate = Validator::make($request->all(), [
            'comment' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }
        if (Auth::user()->admin_role_id == 6) {
            $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $offchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);

            if($offchargeObj->md_approval == 'pending') {
                ThirdPartyOffchargeAndExpunction::where('id', $offchargeObj->id)->update(['md_approval' => 'approved', 'md_comment' => $request['comment']]);
                ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['active_status' => $offchargeObj->type_charges,
                    'ongoing_request' => null]);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->with(['user', 'thirdPartyOutdoorSignDetails',
                    'thirdPartyOffChargeAndExpunction'])->get();

                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'the request has been treated already'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'only MD can approve this request'], $this->errorStatusCode);
        }
    }

    public function mdDisapprovalforRequest(Request $request, $outdoorId, $offchargeId) {
        $validate = Validator::make($request->all(), [
            'comment' => 'required|string'
        ]);

        if($validate->fails()) {
            return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
        }
        if (Auth::user()->admin_role_id == 6) {
            $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $offchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);

            if($offchargeObj->md_approval == null) {
                ThirdPartyOffchargeAndExpunction::where('id', $offchargeObj->id)->update(['md_approval' => 'disapproved', 'hod_recommedation'=> 'disapproved', 'md_comment' => $request['comment'], 'rm_recommedation' => 'disapproved', 'complaince_confirm_status' => 'disapproved']);
                ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['active_status' => null, 'ongoing_request' => null]);
                $getResponse = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->with(['user', 'thirdPartyOutdoorSignDetails',
                    'thirdPartyOffChargeAndExpunction'])->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'the request has been treated already'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'only MD can disapprove this request'], $this->errorStatusCode);
        }
    }

    public function generateNewOffChargeBill($outdoorId, $offchargeId) {

        if((Auth::user()->department == 'Billings') && (Auth::user()->admin_role_id  == 2 || Auth::user()->admin_role_id == 3)) {
                $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
                $offchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);

                if($offchargeObj->md_approval == 'approved') {
                    if($offchargeObj->type_charges == 'expunction') {
                        $billgenerate = [];
                        $billgenerate = self::offchargeAndExpunctionCalculation($outdoorObj->id, $offchargeObj->id, "expunction");
                        ThirdPartyOutdoor::where('id', $outdoorObj->id)->update(['active_status' => 'expunction']);
                        ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id',   $outdoorObj->id)->update(['annual_adsprice_fee'
                                                                    => $billgenerate['adsprice'], 'annual_legal_fee' =>$billgenerate['legalFee'] ]);
                        ThirdPartyOffchargeAndExpunction::where('id', $offchargeObj->id)->update(['offcharge_expunction_bill' => true]);
                        $oldtotal = bcadd($billgenerate['adsprice'], $billgenerate['legalFee'], 2);
                        // Log::info($oldtotal);
                        $oldArears = AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $outdoorObj->id)->latest()->first();
                        $oldArearsBal = $oldArears->balance;
                        $sumoldbal = bcsub($oldArearsBal, $oldtotal, 2);
                        AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $outdoorObj->id)->update(['ads_price' => $billgenerate['adsprice'], 'legal_fee' => $billgenerate['legalFee'],
                                                'price_rate' => $billgenerate['rate'], 'balance' => $sumoldbal]);
                        // return response()->json(['success' => true, 'adsPrice' => $billgenerate['adsprice'], 'legalFee' => $billgenerate['legalFee'], 'rate' =>
                        //                                                     $billgenerate['rate']], $this->successStatusCode);
                    } elseif ($offchargeObj->type_charges == 'off_charge') {
                        $billgenerate = [];
                        $billgenerate = self::offchargeAndExpunctionCalculation($outdoorObj->id, $offchargeObj->id, "off_charge");
                        ThirdPartyOutdoor::where('id', $outdoorObj->id)->update(['active_status' => 'off_charge']);
                        ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id',   $outdoorObj->id)->update(['annual_adsprice_fee'
                        => $billgenerate['adsprice'], 'annual_legal_fee' =>$billgenerate['legalFee'] ]);
                        ThirdPartyOffchargeAndExpunction::where('id', $offchargeObj->id)->update(['offcharge_expunction_bill' => true]);
                        $oldtotal = bcadd($billgenerate['adsprice'], $billgenerate['legalFee'], 2);
                        $oldArears = AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $outdoorObj->id)->latest()->first();
                        $oldArearsBal = $oldArears->balance;
                        $sumoldbal = bcsub($oldArearsBal, $oldtotal, 2);
                        AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $outdoorObj->id)->update(['ads_price' => $billgenerate['adsprice'], 'legal_fee' => $billgenerate['legalFee'],
                                                'price_rate' => $billgenerate['rate'], 'balance' => $sumoldbal]);
                        // return response()->json(['success' => true, 'adsPrice' => $billgenerate['adsprice'], 'legalFee' => $billgenerate['legalFee'], 'rate' =>
                        //     $billgenerate['rate']], $this->successStatusCode);
                    }
                    
                    
                    $startdate = $billgenerate['start'];
                     $enddate = Carbon::parse($billgenerate['adsprice'])->endOfYear();
                     
                     $userdetails = User::where('id', $outdoorObj->user_id)->first();
                     $initial_amount = $billgenerate['initial'];
                    
                     $amountInWord = UtilMethods::getAmountInWords($initial_amount - $oldtotal);
             
                     $commentt = strtoupper($billgenerate['sign']);
                            
                     $data = [
                          'date' => Carbon::now()->format('jS F, Y'),
                          'company' => $userdetails->business_name,
                          'address' => $outdoorObj->address_proposed_site,
                          'email' => $userdetails->email,
                          'cust_id' => $outdoorObj->user_id,
                          'ref_no' => $outdoorObj->payment_id,
                          'user_type' => $userdetails->user_type,
                          'initial_amount_due' => $initial_amount,
                          'debit' => $oldtotal,
                          'credit' => $oldtotal,
                          'note_type' => 'debit',
                          'amount_in_words' => $amountInWord,
                          'startDate' => $startdate->format('jS F, Y'),
                          'endDate' => $enddate->format('jS F, Y'),
                          'comment' => 'ADDITIONAL DEBIT NOTE FOR ' . $commentt . ' RATE'
                          
                     ];
                     
                    
                     $pdf = Pdf::loadView('CreditDebitNote', compact('data'))->setPaper('A3');
                     $fileObj = $pdf->download('CreditDebitNote.pdf');
                    
                      $userObj = User::find($outdoorObj->user_id);
                      $emailcontent = EmailContent::getContent('outdoor_debit_credit_content');
                     $body = $emailcontent;
                     $templateData = [
                         'body' => $body,
                         'name' => $userObj->business_name
                     ];
                     $templateContent = view('hodeoitemplate', $templateData)->render();
                     UtilMethods::debitnotethirdparty($userObj->email, $templateContent, $fileObj);
                     
                     $pdfContent = $pdf->output();
                     $filename = 'BillInvoice_' . now()->format('YmdHis') . '.pdf';
                     $relativePath = 'invoices/' . $filename;
                     $savePath = public_path($relativePath); // Full path to /public/invoices/
                
                     if (!File::exists(public_path('invoices'))) {
                         File::makeDirectory(public_path('invoices'), 0755, true);
                     }
                     file_put_contents($savePath, $pdfContent);
                     $fileUrl = config('app.url') . '/public/' . $relativePath;
                     $adnewrec = new ThirdPartyReceipts();
                     $adnewrec->link = $fileUrl;
                     $adnewrec->type = "Credit / Debit";
                     // $tosaveoutd = ThirdPartyOutdoor::find($getOutDoorObj[$i]['id']);
                     $outdoorObj->thirdpartyreceipts()->save($adnewrec);
                    
                    
                    return response()->json(['success' => true, 'adsPrice' => $billgenerate['adsprice'], 'legalFee' => $billgenerate['legalFee'], 'rate' =>
                            $billgenerate['rate']], $this->successStatusCode);
                    
                } else {
                    return response()->json(['success' => false, 'data' => 'request has not been approved.'], $this->errorStatusCode);
                }
        } else {
            return response()->json(['success' => false, 'data' => 'do not have the priviledge to perform this task'], $this->errorStatusCode);
        }
    }

    public static  function annualOutdoorBills() {

        // $getOutDoorObj = ThirdPartyOutdoor::find($outdoorId);
        // $getOutDoorObj = ThirdPartyOutdoor::whereNotIn('ongoing_request', ['offcharge', 'expunction'])->all();
        $getOutDoorObj = ThirdPartyOutdoor::whereNotIn('ongoing_request', ['offcharge', 'expunction'])->orWhereNull('ongoing_request')
                            ->whereHas('AppHistoryThirdPartyOutdoor', function($query) {
                            $query->whereNotNull('ads_price');})->get()->toArray();
        $getOutDoorObjcount = count($getOutDoorObj) - 1;
        
        // foreach($getOutDoorObj as $outdoors){
        for ($i = 0; $i <= $getOutDoorObjcount; $i++) {
        $year = Carbon::now()->year;
        if((int)$getOutDoorObj[$i]['permit_year'] < (int)$year){
        
        // $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutDoorObj[$i]['id'])->get();

        // $firstDayOfYear = Carbon::createFromDate($year)->startOfYear();
        // $lastDayofYear = Carbon::createFromDate($year)->endOfYear();
        // $numberOfWeekInYear = $lastDayofYear->weeksInYear();
        
        
        
        
        $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id' , $getOutDoorObj[$i]['id'])->first();
        // $getApprovedDate = Carbon::parse($getOutdoorObj->installation_date);
        // $getApprovedDate = Carbon::parse($getOutdoorObj->updated_at);
        $getApprovedDate = Carbon::createFromDate(Carbon::now()->subYear()->year, 12, 01);
                $getPaymentStartDate = $getApprovedDate->addMonth();
        // var_dump($getPaymentStartDate);die();
        // $getPaymentStartDate = $getApprovedDate;

        $lastDayofYear = Carbon::createFromDate(Carbon::now()->year, 12, 31);
        $lastDayofYearLastYear = Carbon::createFromDate(Carbon::now()->year - 1, 12, 31);
        $daysRemaining = $getPaymentStartDate->diffInDays($lastDayofYear);
        $weeksRemaining = $getPaymentStartDate->diffInWeeks($lastDayofYear);
        if($getPaymentStartDate->year == Carbon::now()->year){
        $daysRemainingPYears = 0;
        $weeksRemainingPYears = 0;
        }else{
        $daysRemainingPYears = $getPaymentStartDate->diffInDays($lastDayofYearLastYear);
        $weeksRemainingPYears = $getPaymentStartDate->diffInWeeks($lastDayofYearLastYear);
        }
        $currentYear = Carbon::now()->year;
        $numberDaysInYear = Carbon::createFromDate($currentYear, 12, 31)->dayOfYear;
                // Log::info("Number of days in a year: $numberDaysInYear");
                // Log::info("Number of days remaining: $daysRemaining");
                // Log::info("Last Day of the year: $lastDayofYear");
                // Log::info("Updated app date: $getPaymentStartDate");
                
        $daysRemaining = $daysRemaining + 1;
        if($weeksRemaining < 52){
            $weeksRemaining = $weeksRemaining + 1;
        }
        
        
        // Get pricing from database
        $pricing = OutdoorSignagePricing::where('designation', $getOutdoorDetails->designation)
            ->where('street_category', $getOutdoorDetails->street_category)
            ->where(function($query) use ($getOutdoorDetails) {
                $query->whereNull('dimension_max')
                    ->orWhere('dimension_max', '>=', $getOutdoorDetails->dimension_sign);
            })
            ->where(function($query) use ($getOutdoorDetails) {
                $query->whereNull('dimension_min')
                    ->orWhere('dimension_min', '<=', $getOutdoorDetails->dimension_sign);
            })
            ->first();
    
        if (!$pricing) {
            return response()->json([
                'success' => false, 
                'data' => 'Pricing not configured for this signage type: ' . $getOutdoorDetails->designation
            ], $this->errorStatusCode);
        }
    
        // Calculate price based on dimension
        $dimension = $getOutdoorDetails->dimension_sign;
        $baseRate = $pricing->base_rate;
                
        // Apply dimension multiplier if needed
        if ($pricing->dimension_max && $dimension > $pricing->dimension_max) {
            $baseRate = $baseRate * ($dimension / $pricing->dimension_max) * $pricing->multiplier;
        }
    
        // Calculate daily or weekly rate
        if ($pricing->calculation_type === 'daily') {
            $amountPerDay = $baseRate / $numberDaysInYear;
            $adsPrice = $amountPerDay * $daysRemaining;
            $adsPricePYears = $amountPerDay * $daysRemainingPYears;
        } else {
            // Weekly calculation
            $adsPrice = $baseRate * $weeksRemaining;
            $adsPricePYears = $baseRate * $weeksRemainingPYears;
        }
    
        // Apply side multiplier
        $adsPrice = $adsPrice * $getOutdoorDetails->side;
        $adsPricePYears = $adsPricePYears * $getOutdoorDetails->side;
                
        // Round values
        $adsPrice = round($adsPrice, 2);
        $adsPricePYears = round($adsPricePYears, 2);
                
        // Calculate legal fees
        $legalFee = (1 / 100) * $adsPrice;
        $legalFee = round($legalFee, 2);
        $legalFeePYears = (1 / 100) * $adsPricePYears;
        $legalFeePYears = round($legalFeePYears, 2);
                
        $getRate = $baseRate;
        

        // $adsPrice = 0;
        // $legalFee = 0;
        // $getRate = 0;
        // $adsPricePYears = 0;
        // $legalFeePYears = 0;
        
        
        // if($getOutdoorDetails[0]['designation'] == '16 Sheet') {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 6) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (22400 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 22400;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (44800 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 44800;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (56000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 56000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (140000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 140000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 6) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 6) * 22400;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 22400;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6 ) * 44800;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 44800;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 56000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 56000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 140000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 140000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '48 Sheet') {
        //             if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (84000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 84000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (196000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate =  196000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (280000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 280000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (700000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 700000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 84000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 84000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 196000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 196000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 280000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 280000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 700000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 700000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '96 Sheet') {
        //             if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (168000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 168000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (392000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 392000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1120000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 36) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 168000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 168000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 392000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 392000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 1120000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 2800000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation']== 'Rooftop/Static') {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 36) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1120000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (784000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 784000;
        //                  } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1134000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1134000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1414000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1414000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3094000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3094000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1008000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1008000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1288000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1288000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1568000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1568000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3248000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3248000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1232000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1232000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1512000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1512000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1792000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1792000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3472000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3472000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1456000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1456000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1736000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1736000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2016000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2016000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3696000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3696000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Rooftop/Ultrawave') {
        //             if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1260000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1260000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (4200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1134000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1134000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1554000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1554000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1974000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1974000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (4494000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4494000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1428000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1428000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1848000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1848000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2268000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2268000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (4788000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4788000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1722000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1722000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (2142000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2142000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2562000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2562000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (5082000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5082000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (2016000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2016000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (2422000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2422000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2856000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2856000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (5376000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5376000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole') {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 54) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (980000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 980000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1176000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1176000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1568000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1568000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3920000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3920000;
        //                 }
        //             }elseif($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1400000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2240000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2240000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (5600000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5600000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] >= 108) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1400000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1680000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 2240000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2240000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 5600000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5600000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Front/Back Lit') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 560000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }
                        
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1120000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1120000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (2800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 2800000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '48 Sheet Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (126000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 126000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 126000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 126000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (294000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 294000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 294000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 294000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (420000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 420000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 420000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 420000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1050000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1050000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1050000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1050000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Wall Panel Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1260000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1260000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1260000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1260000;
        //                 }
        //             }  elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (4200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '96 Sheet Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (252000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 252000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 252000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 252000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (588000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 588000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 588000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 588000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (4200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (1820000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1820000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 1820000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1820000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (2184000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2184000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2184000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2184000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (2912000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2912000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2912000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2912000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (7280000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7280000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 7280000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7280000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Wall Drape') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (224 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (224 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (315 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (315 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (385 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (385 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (672 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (672 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Wrap Around' ) {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (175 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (175 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (245 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (245 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (315 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (315 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (546 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (546 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif (str_contains($getOutdoorDetails[0]['designation'], 'LED')) {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 40) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (2600000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2600000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'   ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3250000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3250000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (3900000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3900000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (7800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 80) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (4875000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4875000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'  ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (6175000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 6175000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (7150000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7150000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (14300000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 14300000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 100) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (6825000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 6825000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (8775000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 8775000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (10400000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 10400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (20800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 20800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 100) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (10400000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 10400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (13000000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 13000000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (15600000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate =  15600000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (31200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 31200000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(A)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
        //                         $amountPerDay = (280000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 280000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
        //                         $amountPerDay = (760000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 760000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
        //                         $amountPerDay = (1240000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1240000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
        //                         $amountPerDay = (1720000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1720000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
        //                         $amountPerDay = (2200000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2200000;
        //                 }
        //             }else{
        //                         $amountPerDay = (700000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 700000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(B)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
        //                         $amountPerDay = (390000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 390000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
        //                         $amountPerDay = (1192500 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1192500;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
        //                         $amountPerDay = (1995000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1995000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
        //                         $amountPerDay = (2797500 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2797500;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
        //                         $amountPerDay = (3600000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 3600000;
        //                 }
        //             }else{
        //                         $amountPerDay = (2100000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2100000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(C)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
        //                         $amountPerDay = (500000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 500000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
        //                         $amountPerDay = (1625000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1625000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
        //                         $amountPerDay = (2750000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2750000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
        //                         $amountPerDay = (3375000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 3375000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
        //                         $amountPerDay = (5000000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 5000000;
        //                 }
        //             }else{
        //                         $amountPerDay = (1400000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1400000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(A)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (420000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 420000;
        //             }else{
        //                         $amountPerDay = (1050000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1050000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(B)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (1260000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1260000;
        //             }else{
        //                         $amountPerDay = (3150000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 3150000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(C)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (7000000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 7000000;
        //             }else{
        //                         $amountPerDay = (17500000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 17500000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Iconic Structure (Billboard)') {
        //             $amountPerDay = (14000000 / $numberDaysInYear);
        //             $adsPrice = ($amountPerDay * $daysRemaining);
        //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //             $adsPrice = round($adsPrice, 2);
        //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //             $adsPricePYears = round($adsPricePYears, 2);
        //             $legalFee = (1 / 100) * $adsPrice;
        //             $legalFee = round($legalFee, 2);
        //             $legalFeePYears = (1 / 100) * $adsPricePYears;
        //             $legalFeePYears = round($legalFeePYears, 2);
        //             $getRate = 14000000;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Tank Farm') {
        //             $amountPerDay = (70000 / $numberDaysInYear);
        //             $adsPrice = ($amountPerDay * $daysRemaining);
        //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //             $adsPrice = round($adsPrice, 2);
        //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //             $adsPricePYears = round($adsPricePYears, 2);
        //             $legalFee = (1 / 100) * $adsPrice;
        //             $legalFee = round($legalFee, 2);
        //             $legalFeePYears = (1 / 100) * $adsPricePYears;
        //             $legalFeePYears = round($legalFeePYears, 2);
        //             $getRate = 70000;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Advertisement') {
        //             $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //             $adsPrice = (44800 * $weeksRemaining);
        //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //             $adsPricePYears = (44800 * $weeksRemainingPYears);
        //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //             $legalFeePYears = (1 / 100) * $adsPricePYears;
        //             $legalFee = (1/100) * $adsPrice;
        //             $getRate = 0;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Tank') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (336 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (336 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }else{
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (504 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (504 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Tank Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (420 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (420 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }else{
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (630 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (630 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Bridge Panel (Pedestrian/vehicular)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (1120000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1120000;
        //             }else{
        //                         $amountPerDay = (2800000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2800000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Bridge Panel Ultrawave (Pedestrian/vehicular)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (1680000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1680000;
        //             }else{
        //                         $amountPerDay = (4200000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 4200000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Advert on Bridge built on private sector Initiative') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (560000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 560000;
        //             }else{
        //                         $amountPerDay = (1400000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1400000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Advert on Bridge built on private sector Initiative Ultrawave') {
        //                 $amountPerDay = (840000 / $numberDaysInYear);
        //                 $adsPrice = ($amountPerDay * $daysRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPrice = round($adsPrice, 2);
        //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = round($adsPricePYears, 2);
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $legalFee = round($legalFee, 2);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFeePYears = round($legalFeePYears, 2);
        //                 $getRate = 840000;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole(Neon)') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
        //                     $amountPerDay = (350000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 350000;
        //             } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (672000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 672000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (896000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 896000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2240000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2240000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole(Neon) Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //             } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (728000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 728000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (873600 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 873600;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1164800 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1164800;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2912000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2912000;
        //             }
        //         }

        ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutDoorObj[$i]['id'])->update(['annual_adsprice_fee' => $adsPrice,
                                                                                    'annual_legal_fee' => $legalFee]);
        ThirdPartyOutdoor::where('id', $getOutDoorObj[$i]['id'])->update(['permit_year' => Carbon::now()->year]);
        $totalFee = bcadd($adsPrice, $legalFee, 2);
        $getArrears = AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOutDoorObj[$i]['id'])->latest()->first();
        $getBalance = $getArrears->balance;
        $newBalance = bcadd($totalFee, $getBalance, 2);
        AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOutDoorObj[$i]['id'])->update(['ads_price' => $adsPrice, 'price_rate' => $legalFee,
                                        'price_rate' => $getRate, 'balance' => $newBalance]);
        
        
        
                $getThirdPartyDetails = ThirdPartyOutdoor::where('id', $getOutDoorObj[$i]['id'])->with(['user', 'thirdPartyOutdoorSignDetails'])
                                                ->with(['appHistoryThirdPartyOutdoor' => function($query) {
                                                            $query->latest()->first();
                                                }])->get();
                $userDetails = KycThirdParty::where('user_id', $getThirdPartyDetails[0]['user_id'])->with(['user'])->get();
                $totalFee = bcadd($adsPrice, $legalFee, 2);
                $amountInWord = UtilMethods::getAmountInWords($totalFee);

                $getDate = Carbon::parse($getThirdPartyDetails[0]['updated_at']);
                $getYear = $getDate->year;
                $data = [
                    'custId' => $userDetails[0]['user']['id'],
                    'email' => $userDetails[0]['user']['email'],
                    'userType' => $userDetails[0]['user']['user_type'],
                    'company' => $userDetails[0]['company_name'],
                    'officeAddress' => $userDetails[0]['office_address'],
                    'billYear' => $getYear,
                    'signageLocation' => $getThirdPartyDetails[0]['address_proposed_site'],
                    'appId' => $getThirdPartyDetails[0]['id'],
                    'billStartDate' => Carbon::parse($getThirdPartyDetails[0]['updated_at'])->format('m-d-y'),
                    'billEndDate' => Carbon::now()->endOfYear()->format('m-d-y'),
                    'dimension' => $getThirdPartyDetails[0]['thirdPartyOutdoorSignDetails']['dimension_sign'],
                    'signage' => $getThirdPartyDetails[0]['thirdPartyOutdoorSignDetails']['designation'],
                    'zone' => $getThirdPartyDetails[0]['thirdPartyOutdoorSignDetails']['street_category'],
                    'side' => $getThirdPartyDetails[0]['thirdPartyOutdoorSignDetails']['side'],
                    'rate' => $getThirdPartyDetails[0]['appHistoryThirdPartyOutdoor'][0]['price_rate'],
                    'adsFee' => $getThirdPartyDetails[0]['appHistoryThirdPartyOutdoor'][0]['ads_price'],
                    'legalFee' => $getThirdPartyDetails[0]['appHistoryThirdPartyOutdoor'][0]['legal_fee'],
                    'totalFee' => $totalFee,
                    'amountInWord' => $amountInWord
                ];


                $pdf = Pdf::loadView('BillInvoice', compact('data'))->setPaper('A3');
                $fileObj = $pdf->download('BillInvoice.pdf');

                $getUserObj = User::find($userDetails[0]['user']['id']);
                $emailcontent = EmailContent::getContent('outdoor_bill_content');

                $body = $emailcontent;
                $templateData = [
                    'body' => $body,
                    'name' => $getUserObj->company_name
                ];
                $templateContent = view('hodeoitemplate', $templateData)->render();
                UtilMethods::outdoorSignageInvoiceBill($getUserObj->email, $templateContent, $fileObj);
                
                $pdfContent = $pdf->output();
                $filename = 'BillInvoice_' . now()->format('YmdHis') . '.pdf';
                $relativePath = 'invoices/' . $filename;
                $savePath = public_path($relativePath); // Full path to /public/invoices/
                
                if (!File::exists(public_path('invoices'))) {
                    File::makeDirectory(public_path('invoices'), 0755, true);
                }
                file_put_contents($savePath, $pdfContent);
                $fileUrl = config('app.url') . '/public/' . $relativePath;
                $adnewrec = new ThirdPartyReceipts();
                $adnewrec->link = $fileUrl;
                $adnewrec->type = "Invoice";
                $tosaveoutd = ThirdPartyOutdoor::find($getOutDoorObj[$i]['id']);
                $tosaveoutd->thirdpartyreceipts()->save($adnewrec);
        }
            
        }
    }

    public  static function offchargeAndExpunctionCalculation($outdoorId, $offchargeId, $type) {
        
        if($type == "expunction"){

        // $getOutDoorObj = ThirdPartyOutdoor::find($outdoorId);
        // $offchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);
        // $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutDoorObj->id)->get();

        // // $getLastPaymentDay = Carbon::parse($offchargeObj->created_at);
        // $getLastPaymentDay = Carbon::now();
        // $year = $getLastPaymentDay->year;
        // $firstDayOfYear = Carbon::createFromDate($year)->startOfYear();
        // $dayDiff = $firstDayOfYear->diffInDays($getLastPaymentDay);
        // $weekDiff = $firstDayOfYear->diffInWeeks($getLastPaymentDay);
        // $numberDaysInYear = Carbon::createFromDate($year, 12, 31)->dayOfYear;

        // $lastDayofYear = Carbon::createFromDate($year)->endOfYear();
        // $numberOfWeekInYear = $lastDayofYear->weeksInYear();
        
        $getOutDoorObj = ThirdPartyOutdoor::find($outdoorId);
        $offchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);
        // $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutDoorObj->id)->get();

        // $getLastPaymentDay = Carbon::parse($offchargeObj->expunge_date);
        // // $getLastPaymentDay = Carbon::now();
        // $year = $getLastPaymentDay->year;
        // $firstDayOfYear = Carbon::createFromDate($year)->startOfYear();
        // $lastDayofYear = Carbon::createFromDate($year)->endOfYear();
        // $dayDiff = $getLastPaymentDay->diffInDays($lastDayofYear);
        // $weekDiff = $getLastPaymentDay->diffInWeeks($lastDayofYear);
        // $numberDaysInYear = Carbon::createFromDate($year, 12, 31)->dayOfYear;
        // $numberOfWeekInYear = $lastDayofYear->weeksInYear();
        
        
        
                $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id' , $getOutDoorObj->id)->first();
                // $getApprovedDate = Carbon::parse($getOutdoorObj->installation_date);
                $getApprovedDate = Carbon::parse($offchargeObj->expunge_date);
                // $getApprovedDate = Carbon::createFromDate(2023, 12, 01);
                $getPaymentStartDate = $getApprovedDate;
                // var_dump($getPaymentStartDate);die();
                // $getPaymentStartDate = $getApprovedDate;

                $lastDayofYear = Carbon::createFromDate(Carbon::now()->year, 12, 31);
                $lastDayofYearLastYear = Carbon::createFromDate(Carbon::now()->year - 1, 12, 31);
                $daysRemaining = $getPaymentStartDate->diffInDays($lastDayofYear);
                $weeksRemaining = $getPaymentStartDate->diffInWeeks($lastDayofYear);
                if($getPaymentStartDate->year == Carbon::now()->year){
                $daysRemainingPYears = 0;
                $weeksRemainingPYears = 0;
                }else{
                $daysRemainingPYears = $getPaymentStartDate->diffInDays($lastDayofYearLastYear);
                $weeksRemainingPYears = $getPaymentStartDate->diffInWeeks($lastDayofYearLastYear);
                }
                $currentYear = Carbon::now()->year;
                $numberDaysInYear = Carbon::createFromDate($currentYear, 12, 31)->dayOfYear;
                
                $daysRemaining = $daysRemaining + 1;
                if($weeksRemaining < 52){
                    $weeksRemaining = $weeksRemaining + 1;
                }
            
        }elseif($type == "off_charge"){

        $getOutDoorObj = ThirdPartyOutdoor::find($outdoorId);
        $offchargeObj = ThirdPartyOffchargeAndExpunction::find($offchargeId);
        // $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutDoorObj->id)->get();

        // $getLastPaymentDay = Carbon::parse($offchargeObj->off_charge_start);
        // // $getLastPaymentDay = Carbon::now();
        // $year = $getLastPaymentDay->year;
        // $firstDayOfYear = Carbon::createFromDate($year)->startOfYear();
        // // $lastDayofYear = Carbon::createFromDate($year)->endOfYear();
        // $lastDayofYear = Carbon::parse($offchargeObj->off_charge_end);
        // $dayDiff = $getLastPaymentDay->diffInDays($lastDayofYear);
        // $weekDiff = $getLastPaymentDay->diffInWeeks($lastDayofYear);
        // $numberDaysInYear = Carbon::createFromDate($year, 12, 31)->dayOfYear;
        // $numberOfWeekInYear = $lastDayofYear->weeksInYear();
        
        
                $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id' , $getOutDoorObj->id)->first();
                // $getApprovedDate = Carbon::parse($getOutdoorObj->installation_date);
                $getApprovedDate = Carbon::parse($offchargeObj->off_charge_start);
                // $getApprovedDate = Carbon::createFromDate(2023, 12, 01);
                $getPaymentStartDate = $getApprovedDate;
                // var_dump($getPaymentStartDate);die();
                // $getPaymentStartDate = $getApprovedDate;

                $lastDayofYear = Carbon::parse($offchargeObj->off_charge_end);
                $lastDayofYearLastYear = Carbon::parse($offchargeObj->off_charge_end);
                $daysRemaining = $getPaymentStartDate->diffInDays($lastDayofYear);
                $weeksRemaining = $getPaymentStartDate->diffInWeeks($lastDayofYear);
                if($getPaymentStartDate->year == Carbon::now()->year){
                $daysRemainingPYears = 0;
                $weeksRemainingPYears = 0;
                }else{
                $daysRemainingPYears = $getPaymentStartDate->diffInDays($lastDayofYearLastYear);
                $weeksRemainingPYears = $getPaymentStartDate->diffInWeeks($lastDayofYearLastYear);
                }
                $currentYear = Carbon::now()->year;
                $numberDaysInYear = Carbon::createFromDate($currentYear, 12, 31)->dayOfYear;
                
                $daysRemaining = $daysRemaining + 1;
                if($weeksRemaining < 52){
                    $weeksRemaining = $weeksRemaining + 1;
                }
            
        }
        
        
                // Get pricing from database
        $pricing = OutdoorSignagePricing::where('designation', $getOutdoorDetails->designation)
            ->where('street_category', $getOutdoorDetails->street_category)
            ->where(function($query) use ($getOutdoorDetails) {
                $query->whereNull('dimension_max')
                    ->orWhere('dimension_max', '>=', $getOutdoorDetails->dimension_sign);
            })
            ->where(function($query) use ($getOutdoorDetails) {
                $query->whereNull('dimension_min')
                    ->orWhere('dimension_min', '<=', $getOutdoorDetails->dimension_sign);
            })
            ->first();
    
        if (!$pricing) {
            return response()->json([
                'success' => false, 
                'data' => 'Pricing not configured for this signage type: ' . $getOutdoorDetails->designation
            ], $this->errorStatusCode);
        }
    
        // Calculate price based on dimension
        $dimension = $getOutdoorDetails->dimension_sign;
        $baseRate = $pricing->base_rate;
                
        // Apply dimension multiplier if needed
        if ($pricing->dimension_max && $dimension > $pricing->dimension_max) {
            $baseRate = $baseRate * ($dimension / $pricing->dimension_max) * $pricing->multiplier;
        }
    
        // Calculate daily or weekly rate
        if ($pricing->calculation_type === 'daily') {
            $amountPerDay = $baseRate / $numberDaysInYear;
            $adsPrice = $amountPerDay * $daysRemaining;
            $adsPricePYears = $amountPerDay * $daysRemainingPYears;
        } else {
            // Weekly calculation
            $adsPrice = $baseRate * $weeksRemaining;
            $adsPricePYears = $baseRate * $weeksRemainingPYears;
        }
    
        // Apply side multiplier
        $adsPrice = $adsPrice * $getOutdoorDetails->side;
        $adsPricePYears = $adsPricePYears * $getOutdoorDetails->side;
                
        // Round values
        $adsPrice = round($adsPrice, 2);
        $adsPricePYears = round($adsPricePYears, 2);
                
        // Calculate legal fees
        $legalFee = (1 / 100) * $adsPrice;
        $legalFee = round($legalFee, 2);
        $legalFeePYears = (1 / 100) * $adsPricePYears;
        $legalFeePYears = round($legalFeePYears, 2);
                
        $getRate = $baseRate;

        // $adsPrice = 0;
        // $legalFee = 0;
        // $getRate = 0;
        // $adsPricePYears = 0;
        // $legalFeePYears = 0;
        
        // if($getOutdoorDetails[0]['designation'] == '16 Sheet') {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 6) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (22400 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 22400;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (44800 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 44800;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (56000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 56000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (140000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 140000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 6) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 6) * 22400;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 22400;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6 ) * 44800;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 44800;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 56000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 56000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 140000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 140000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '48 Sheet') {
        //             if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (84000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 84000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (196000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate =  196000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (280000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 280000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (700000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 700000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 84000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 84000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 196000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 196000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 280000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 280000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 700000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 700000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '96 Sheet') {
        //             if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (168000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 168000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (392000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 392000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1120000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 36) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 168000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 168000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 392000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 392000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 1120000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 2800000;
        //                     $amountPerDay = ($getYearRate / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation']== 'Rooftop/Static') {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 36) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1120000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (784000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 784000;
        //                  } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1134000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1134000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1414000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1414000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3094000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3094000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1008000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1008000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1288000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1288000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1568000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1568000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3248000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3248000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1232000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1232000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1512000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1512000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1792000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1792000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3472000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3472000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1456000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1456000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1736000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1736000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2016000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2016000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3696000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3696000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Rooftop/Ultrawave') {
        //             if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1260000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1260000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (4200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1134000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1134000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1554000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1554000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1974000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1974000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (4494000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4494000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1428000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1428000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1848000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1848000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2268000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2268000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (4788000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4788000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1722000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1722000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (2142000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2142000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2562000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2562000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (5082000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5082000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (2016000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2016000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (2422000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2422000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2856000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2856000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (5376000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5376000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole') {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 54) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (980000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 980000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1176000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1176000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1568000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1568000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3920000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3920000;
        //                 }
        //             }elseif($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (1400000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (2240000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2240000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (5600000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5600000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] >= 108) {
        //                 if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1400000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1680000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 2240000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2240000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 5600000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 5600000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Front/Back Lit') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 560000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }
                        
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1120000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1120000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1120000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (2800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 2800000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2800000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '48 Sheet Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (126000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 126000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 126000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 126000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (294000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 294000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 294000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 294000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (420000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 420000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 420000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 420000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1050000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1050000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1050000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1050000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Wall Panel Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (840000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 840000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1260000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1260000;
        //                 }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1260000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1260000;
        //                 }
        //             }  elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (4200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == '96 Sheet Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (252000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 252000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 252000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 252000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (588000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 588000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 588000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 588000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (1680000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1680000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
        //                     $amountPerDay = (4200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4200000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (1820000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1820000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 1820000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1820000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (2184000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2184000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2184000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2184000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (2912000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2912000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2912000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2912000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
        //                     $amountPerDay = (7280000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7280000;
        //                 }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
        //                     $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 7280000) / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7280000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Wall Drape') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (224 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (224 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (315 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (315 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (385 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (385 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (672 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (672 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Wrap Around' ) {
        //             if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (175 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (175 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (245 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (245 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (315 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (315 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (546 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (546 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif (str_contains($getOutdoorDetails[0]['designation'], 'LED')) {
        //             if($getOutdoorDetails[0]['dimension_sign'] <= 40) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (2600000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2600000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'   ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (3250000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3250000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (3900000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 3900000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (7800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 80) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (4875000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 4875000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'  ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (6175000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 6175000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (7150000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 7150000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (14300000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 14300000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 100) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (6825000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 6825000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (8775000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 8775000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (10400000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 10400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (20800000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 20800000;
        //                 }
        //             } elseif ($getOutdoorDetails[0]['dimension_sign'] > 100) {
        //                 if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (10400000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 10400000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (13000000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 13000000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
        //                     $getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (15600000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate =  15600000;
        //                 } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
        //                     $amountPerDay = (31200000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                     $adsPrice = round($adsPrice, 2);
        //                     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                     $adsPricePYears = round($adsPricePYears, 2);
        //                     $legalFee = (1 / 100) * $adsPrice;
        //                     $legalFee = round($legalFee, 2);
        //                     $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                     $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 31200000;
        //                 }
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(A)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
        //                         $amountPerDay = (280000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 280000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
        //                         $amountPerDay = (760000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 760000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
        //                         $amountPerDay = (1240000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1240000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
        //                         $amountPerDay = (1720000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1720000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
        //                         $amountPerDay = (2200000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2200000;
        //                 }
        //             }else{
        //                         $amountPerDay = (700000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 700000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(B)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
        //                         $amountPerDay = (390000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 390000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
        //                         $amountPerDay = (1192500 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1192500;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
        //                         $amountPerDay = (1995000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1995000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
        //                         $amountPerDay = (2797500 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2797500;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
        //                         $amountPerDay = (3600000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 3600000;
        //                 }
        //             }else{
        //                         $amountPerDay = (2100000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2100000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(C)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
        //                         $amountPerDay = (500000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 500000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
        //                         $amountPerDay = (1625000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1625000;
                            
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
        //                         $amountPerDay = (2750000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2750000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
        //                         $amountPerDay = (3375000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 3375000;
        //                 } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
        //                         $amountPerDay = (5000000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 5000000;
        //                 }
        //             }else{
        //                         $amountPerDay = (1400000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1400000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(A)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (420000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 420000;
        //             }else{
        //                         $amountPerDay = (1050000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1050000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(B)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (1260000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1260000;
        //             }else{
        //                         $amountPerDay = (3150000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 3150000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(C)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (7000000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 7000000;
        //             }else{
        //                         $amountPerDay = (17500000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 17500000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Iconic Structure (Billboard)') {
        //             $amountPerDay = (14000000 / $numberDaysInYear);
        //             $adsPrice = ($amountPerDay * $daysRemaining);
        //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //             $adsPrice = round($adsPrice, 2);
        //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //             $adsPricePYears = round($adsPricePYears, 2);
        //             $legalFee = (1 / 100) * $adsPrice;
        //             $legalFee = round($legalFee, 2);
        //             $legalFeePYears = (1 / 100) * $adsPricePYears;
        //             $legalFeePYears = round($legalFeePYears, 2);
        //             $getRate = 14000000;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Tank Farm') {
        //             $amountPerDay = (70000 / $numberDaysInYear);
        //             $adsPrice = ($amountPerDay * $daysRemaining);
        //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //             $adsPrice = round($adsPrice, 2);
        //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //             $adsPricePYears = round($adsPricePYears, 2);
        //             $legalFee = (1 / 100) * $adsPrice;
        //             $legalFee = round($legalFee, 2);
        //             $legalFeePYears = (1 / 100) * $adsPricePYears;
        //             $legalFeePYears = round($legalFeePYears, 2);
        //             $getRate = 70000;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Advertisement') {
        //             $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //             $adsPrice = (44800 * $weeksRemaining);
        //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //             $adsPricePYears = (44800 * $weeksRemainingPYears);
        //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //             $legalFeePYears = (1 / 100) * $adsPricePYears;
        //             $legalFee = (1/100) * $adsPrice;
        //             $getRate = 0;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Tank') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (336 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (336 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }else{
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (504 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (504 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Tank Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (420 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (420 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }else{
        //                 $getDimension = $getOutdoorDetails[0]['dimension_sign'];
        //                 $adsPrice = (630 * $getDimension * $weeksRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = (630 * $getDimension * $weeksRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFee = (1/100) * $adsPrice;
        //                 $getRate = 0;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Bridge Panel (Pedestrian/vehicular)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (1120000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1120000;
        //             }else{
        //                         $amountPerDay = (2800000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 2800000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Bridge Panel Ultrawave (Pedestrian/vehicular)') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (1680000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1680000;
        //             }else{
        //                         $amountPerDay = (4200000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 4200000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Advert on Bridge built on private sector Initiative') {
        //             if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
        //                         $amountPerDay = (560000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 560000;
        //             }else{
        //                         $amountPerDay = (1400000 / $numberDaysInYear);
        //                         $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                         $getRate = 1400000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Advert on Bridge built on private sector Initiative Ultrawave') {
        //                 $amountPerDay = (840000 / $numberDaysInYear);
        //                 $adsPrice = ($amountPerDay * $daysRemaining);
        //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                 $adsPrice = round($adsPrice, 2);
        //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                 $adsPricePYears = round($adsPricePYears, 2);
        //                 $legalFee = (1 / 100) * $adsPrice;
        //                 $legalFee = round($legalFee, 2);
        //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                 $legalFeePYears = round($legalFeePYears, 2);
        //                 $getRate = 840000;
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole(Neon)') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
        //                     $amountPerDay = (350000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 350000;
        //             } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (672000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 672000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (896000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 896000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2240000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2240000;
        //             }
        // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole(Neon) Ultrawave') {
        //             if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
        //                     $amountPerDay = (560000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 560000;
        //             } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
        //                     $amountPerDay = (728000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 728000;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
        //                     $amountPerDay = (873600 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 873600;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
        //                     $amountPerDay = (1164800 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 1164800;
        //             } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
        //                     $amountPerDay = (2912000 / $numberDaysInYear);
        //                     $adsPrice = ($amountPerDay * $daysRemaining);
        //                         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
        //                         $adsPrice = round($adsPrice, 2);
        //                         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
        //                         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
        //                         $adsPricePYears = round($adsPricePYears, 2);
        //                         $legalFee = (1 / 100) * $adsPrice;
        //                         $legalFee = round($legalFee, 2);
        //                         $legalFeePYears = (1 / 100) * $adsPricePYears;
        //                         $legalFeePYears = round($legalFeePYears, 2);
        //                     $getRate = 2912000;
        //             }
        //         }

        $billGenerated = [
            'adsprice' => $adsPrice,
            'legalFee' => $legalFee,
            'rate' => $getRate,
            'start' => $getApprovedDate,
            'initial' => $getOutdoorDetails->ads_price + $getOutdoorDetails->legal_fee,
            'sign' => $getOutdoorDetails->designation
        ];

        return $billGenerated;
    }

    ////////////////////UPGRADE AND DOWNGRADE ADMIN OPERATION////////////////////////////////////////////////////////////////////

    public function setSiteInspectionStatus($outdoorId, $requestId) {
        if (Auth::user()->department == 'Registry' && (Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3)) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            if($outdoorObj->assigned_officer == (string) Auth::user()->id) {
                $upgradeDowngradeObj  =  ThirdPartyUpgradeAndDowngrade::find($requestId);
                if($upgradeDowngradeObj->site_inspection_status == 'pending') {
                    ThirdPartyUpgradeAndDowngrade::where('id', $upgradeDowngradeObj->id)->update(['site_inspection_status' => 'done']);
                    $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeDowngradeObj->id)->get();
                    return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'site inspection status is not pending'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'you are not the assigned rm to this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an rm or do not belong to regsitry department'], $this->errorStatusCode);
        }
    }

    public function assessmentReportByRM($outdoorId, $requestId, Request $request) {
        if (Auth::user()->department == 'Registry' && (Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3)) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            if($outdoorObj->assigned_officer == (string) Auth::user()->id) {
                $validate = Validator::make($request->all(), [
                    'reportFile' => 'required|string',
                    'installation_date' => 'required|date'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
                $assessmentReport = $request->all();
                if($upgradeandDowngradeObj->first_assesment == null) {

                    $utilObj = new UtilMethods();
                    $rmreport = $utilObj->convertBase64toPdfV2($assessmentReport['reportFile']);
                    ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['first_assesment' => $rmreport, 'installation_date' => $assessmentReport['installation_date'], 'hod_recommendation' => 'pending']);
                    $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                    return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'assessment report already uploaded'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'you are not the assigned rm to this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an rm or do not belong to regsitry department'], $this->errorStatusCode);
        }
    }

    public function assessmentReportByHOU($outdoorId, $requestId, Request $request) {
        if (Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
                $validate = Validator::make($request->all(), [
                    'reportFile' => 'required|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
                $assessmentReport = $request->all();
                if($upgradeandDowngradeObj->second_assessment == null) {

                    $utilObj = new UtilMethods();
                    $rmreport = $utilObj->convertBase64toPdfV2($assessmentReport['reportFile']);
                    ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['second_assessment' => $rmreport,
                                                                                                'hod_recommendation' => 'pending']);
                    $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                    return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
                } else {
                    return response()->json(['success' => false, 'data' => 'assessment report already uploaded'], $this->errorStatusCode);
                }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }

    public function recommendUpgradeandDowngradeApproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }

        if (Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
            if($upgradeandDowngradeObj->hod_recommendation == 'pending') {
                ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['hod_recommendation' => 'approved',
                                                                                           'md_approval' => 'pending', 'hod_comment' => $request['comment']]);
                $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'Hod has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an Hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }

    public function recommendUpgradeandDowngradeDisapproval($outdoorId, $requestId) {

        if (Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
            if($upgradeandDowngradeObj->hod_recommendation == 'pending') {
                ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['hod_recommendation' => 'disapproved',
                                                                                                                    'md_approval' => 'pending']);
                $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'Hod has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an Hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }

    public function mdUpgradeandDowngradeApproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                
        if (Auth::user()->admin_role_id == 6) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
            if($upgradeandDowngradeObj->md_approval == 'pending') {
                $designation = $upgradeandDowngradeObj->request_outdoor_signage;
                ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['md_approval' => 'approved', 'md_comment' => $request['comment']]);
                // ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $outdoorObj->id)->update(['designation' => $designation]);
                // ThirdPartyOutdoor::where('id', $outdoorObj->id)->update(['active_status' => $upgradeandDowngradeObj->type_charges,
                //     'ongoing_request' => null]);
                ThirdPartyOutdoor::where('id', $outdoorObj->id)->update(['active_status' => $upgradeandDowngradeObj->type_charges]);
                $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'MD has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you need md priviledge to take this action'], $this->errorStatusCode);
        }
    }

    public function mdUpgradeandDowngradeDisapproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                
        if (Auth::user()->admin_role_id == 6) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
            if($upgradeandDowngradeObj->md_approval == 'pending') {
                ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['md_approval' => 'disapproved',
                    'ongoing_request' => null, 'md_comment' => $request['comment']]);
                $response = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'MD has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you need md priviledge to take this action'], $this->errorStatusCode);
        }
    }

   public function billCalcforUpgradeandDowngrade(Request $request, $outdoorId, $requestId) {

       if((Auth::user()->department == 'Billings') && (Auth::user()->admin_role_id  == 2 || Auth::user()->admin_role_id == 3)) {
          $validate = Validator::make($request->all(), [
                'streetClassification' => 'required|string',
                'price_rate' => 'nullable|numeric|between:0,99999999.99'
            ]);


            if($validate->fails()) {
                return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
            }

            $billingInput  = $request->all();
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeAndDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);
            $upgradedowngradeadd = $upgradeAndDowngradeObj->request_outdoor_signage;
            $outdoorObj->type_sign = $upgradedowngradeadd;
            $outdoorObj->save();

            if($upgradeAndDowngradeObj->md_approval == 'approved' && ($upgradeAndDowngradeObj->bill_status == null ||
                    $upgradeAndDowngradeObj->bill_status == 'pending') ) {
                    $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $outdoorObj->id)->first();
                    $initial_amount = $getOutdoorDetails->ads_price + $getOutdoorDetails->legal_fee;

                    
                    
                    
                    
                    // $getOutdoorDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id' , $outdoorObj->id)->get();
                    // $getApprovedDate = Carbon::parse($getOutdoorObj->installation_date);
                    $getApprovedDate = Carbon::parse($upgradeAndDowngradeObj->installation_date);
                    // $getApprovedDate = Carbon::createFromDate(2023, 12, 01);
                    $getPaymentStartDate = $getApprovedDate;
                    // var_dump($getPaymentStartDate);die();
                    // $getPaymentStartDate = $getApprovedDate;
    
                    $lastDayofYear = Carbon::createFromDate(Carbon::now()->year, 12, 31);
                    $lastDayofYearLastYear = Carbon::createFromDate(Carbon::now()->year - 1, 12, 31);
                    $daysRemaining = $getPaymentStartDate->diffInDays($lastDayofYear);
                    $weeksRemaining = $getPaymentStartDate->diffInWeeks($lastDayofYear);
                    if($getPaymentStartDate->year == Carbon::now()->year){
                    $daysRemainingPYears = 0;
                    $weeksRemainingPYears = 0;
                    }else{
                    $daysRemainingPYears = $getPaymentStartDate->diffInDays($lastDayofYearLastYear);
                    $weeksRemainingPYears = $getPaymentStartDate->diffInWeeks($lastDayofYearLastYear);
                    }
                    $currentYear = Carbon::now()->year;
                    $numberDaysInYear = Carbon::createFromDate($currentYear, 12, 31)->dayOfYear;
                    // Log::info("Number of days in a year: $numberDaysInYear");
                    // Log::info("Number of days remaining: $daysRemaining");
                    // Log::info("Last Day of the year: $lastDayofYear");
                    // Log::info("Updated app date: $getPaymentStartDate");
                    
                    $daysRemaining = $daysRemaining + 1;
                    if($weeksRemaining < 52){
                        $weeksRemaining = $weeksRemaining + 1;
                    }
                    
                    
                    
    
                    // $adsPrice = 0;
                    // $legalFee = 0;
                    // $getRate = 0;
                    // $adsPricePYears = 0;
                    // $legalFeePYears = 0;
                    
                    // $adsPriceRemove = 0;
                    // $legalFeeRemove = 0;
                    // $adsPriceRemovePYears = 0;
                    // $legalFeeRemovePYears = 0;
                    
                    // if($getOutdoorDetails[0]['designation'] == '16 Sheet') {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 6) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (22400 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 22400;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (44800 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 44800;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (56000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 56000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (140000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 140000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 6) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 6) * 22400;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 22400;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6 ) * 44800;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 44800;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 56000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 56000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 140000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 140000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == '48 Sheet') {
                    //     if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (84000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 84000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (196000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate =  196000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (280000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 280000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (700000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 700000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 84000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 84000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 196000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 196000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 280000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 280000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 700000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 700000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == '96 Sheet') {
                    //     if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (168000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 168000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (392000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 392000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1120000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1120000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2800000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 36) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 168000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 168000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 392000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 392000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 1120000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1120000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 2800000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation']== 'Rooftop/Static') {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 36) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 560000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 840000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1120000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1120000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2800000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (784000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 784000;
                    //          } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1134000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1134000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1414000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1414000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3094000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3094000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1008000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1008000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1288000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1288000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1568000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1568000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3248000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3248000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1232000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1232000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1512000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1512000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1792000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1792000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3472000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3472000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1456000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1456000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1736000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1736000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2016000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2016000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3696000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3696000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Rooftop/Ultrawave') {
                    //     if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 840000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1260000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1260000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (4200000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4200000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1134000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1134000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1554000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1554000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1974000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1974000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (4494000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4494000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1428000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1428000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1848000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1848000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2268000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2268000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (4788000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4788000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1722000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1722000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (2142000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2142000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2562000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2562000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (5082000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 5082000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (2016000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2016000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (2422000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2422000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2856000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2856000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (5376000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 5376000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole') {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 54) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (980000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 980000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1176000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1176000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1568000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1568000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3920000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3920000;
                    //         }
                    //     }elseif($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1400000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2240000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2240000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (5600000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 5600000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] >= 108) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1400000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1680000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 2240000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2240000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 5600000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 5600000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Front/Back Lit') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 560000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 560000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 560000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 840000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 840000;
                    //         }
                            
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1120000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1120000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1120000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1120000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (2800000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2800000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 2800000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == '48 Sheet Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (126000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 126000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 126000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 126000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (294000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 294000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 294000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 294000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (420000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 420000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 420000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 420000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1050000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1050000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1050000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1050000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Wall Panel Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 840000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 840000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1260000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1260000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1260000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1260000;
                    //         }
                    //     }  elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (4200000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4200000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4200000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == '96 Sheet Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (252000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 252000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 252000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 252000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (588000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 588000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 588000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 588000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1680000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (4200000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4200000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4200000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (1820000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1820000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 1820000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1820000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (2184000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2184000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2184000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2184000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (2912000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2912000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2912000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2912000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (7280000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 7280000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 7280000) / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 7280000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Wall Drape') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (224 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (224 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (315 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (315 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (385 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (385 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (672 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (672 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Wrap Around' ) {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (175 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (175 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1 / 100) * $adsPrice;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (245 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (245 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1 / 100) * $adsPrice;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (315 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (315 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1 / 100) * $adsPrice;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (546 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (546 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1 / 100) * $adsPrice;
                    //         $getRate = 0;
                    //     }
                    // } elseif (str_contains($getOutdoorDetails[0]['designation'], 'LED')) {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 40) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (2600000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2600000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'   ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3250000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3250000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (3900000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 3900000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (7800000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 7800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 80) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (4875000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 4875000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'  ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (6175000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 6175000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (7150000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 7150000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (14300000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 14300000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 100) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (6825000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 6825000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (8775000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 8775000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (10400000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 10400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (20800000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 20800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 100) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (10400000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 10400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (13000000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 13000000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (15600000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate =  15600000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (31200000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //             $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //             $adsPrice = round($adsPrice, 2);
                    //             $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPricePYears = round($adsPricePYears, 2);
                    //             $legalFee = (1 / 100) * $adsPrice;
                    //             $legalFee = round($legalFee, 2);
                    //             $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //             $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 31200000;
                    //         }
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(A)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
                    //                 $amountPerDay = (280000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 280000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
                    //                 $amountPerDay = (760000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 760000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
                    //                 $amountPerDay = (1240000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1240000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
                    //                 $amountPerDay = (1720000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1720000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
                    //                 $amountPerDay = (2200000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 2200000;
                    //         }
                    //     }else{
                    //                 $amountPerDay = (700000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 700000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(B)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
                    //                 $amountPerDay = (390000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 390000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
                    //                 $amountPerDay = (1192500 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1192500;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
                    //                 $amountPerDay = (1995000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1995000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
                    //                 $amountPerDay = (2797500 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 2797500;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
                    //                 $amountPerDay = (3600000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 3600000;
                    //         }
                    //     }else{
                    //                 $amountPerDay = (2100000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 2100000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs(C)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
                    //                 $amountPerDay = (500000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 500000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
                    //                 $amountPerDay = (1625000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1625000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
                    //                 $amountPerDay = (2750000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 2750000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
                    //                 $amountPerDay = (3375000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 3375000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
                    //                 $amountPerDay = (5000000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 5000000;
                    //         }
                    //     }else{
                    //                 $amountPerDay = (1400000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1400000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(A)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (420000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 420000;
                    //     }else{
                    //                 $amountPerDay = (1050000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1050000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(B)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (1260000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1260000;
                    //     }else{
                    //                 $amountPerDay = (3150000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 3150000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Overhead Gantries/Directional signs Ultrawave(C)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (7000000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 7000000;
                    //     }else{
                    //                 $amountPerDay = (17500000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 17500000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Iconic Structure (Billboard)') {
                    //     $amountPerDay = (14000000 / $numberDaysInYear);
                    //     $adsPrice = ($amountPerDay * $daysRemaining);
                    //     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //     $adsPrice = round($adsPrice, 2);
                    //     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //     $adsPricePYears = round($adsPricePYears, 2);
                    //     $legalFee = (1 / 100) * $adsPrice;
                    //     $legalFee = round($legalFee, 2);
                    //     $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //     $legalFeePYears = round($legalFeePYears, 2);
                    //     $getRate = 14000000;
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Tank Farm') {
                    //     $amountPerDay = (70000 / $numberDaysInYear);
                    //     $adsPrice = ($amountPerDay * $daysRemaining);
                    //     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //     $adsPrice = round($adsPrice, 2);
                    //     $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //     $adsPricePYears = round($adsPricePYears, 2);
                    //     $legalFee = (1 / 100) * $adsPrice;
                    //     $legalFee = round($legalFee, 2);
                    //     $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //     $legalFeePYears = round($legalFeePYears, 2);
                    //     $getRate = 70000;
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Advertisement') {
                    //     $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //     $adsPrice = (44800 * $weeksRemaining);
                    //     $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //     $adsPricePYears = (44800 * $weeksRemainingPYears);
                    //     $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //     $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //     $legalFee = (1/100) * $adsPrice;
                    //     $getRate = 0;
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Tank') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (336 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (336 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     }else{
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (504 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (504 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Water Tank Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (420 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (420 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     }else{
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPrice = (630 * $getDimension * $weeksRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = (630 * $getDimension * $weeksRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFee = (1/100) * $adsPrice;
                    //         $getRate = 0;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Bridge Panel (Pedestrian/vehicular)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (1120000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1120000;
                    //     }else{
                    //                 $amountPerDay = (2800000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 2800000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Bridge Panel Ultrawave (Pedestrian/vehicular)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (1680000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1680000;
                    //     }else{
                    //                 $amountPerDay = (4200000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 4200000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Advert on Bridge built on private sector Initiative') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (560000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 560000;
                    //     }else{
                    //                 $amountPerDay = (1400000 / $numberDaysInYear);
                    //                 $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //                 $getRate = 1400000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Advert on Bridge built on private sector Initiative Ultrawave') {
                    //         $amountPerDay = (840000 / $numberDaysInYear);
                    //         $adsPrice = ($amountPerDay * $daysRemaining);
                    //         $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //         $adsPrice = round($adsPrice, 2);
                    //         $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //         $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //         $adsPricePYears = round($adsPricePYears, 2);
                    //         $legalFee = (1 / 100) * $adsPrice;
                    //         $legalFee = round($legalFee, 2);
                    //         $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //         $legalFeePYears = round($legalFeePYears, 2);
                    //         $getRate = 840000;
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole(Neon)') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
                    //             $amountPerDay = (350000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 350000;
                    //     } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 560000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (672000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 672000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (896000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 896000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2240000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2240000;
                    //     }
                    // } elseif ($getOutdoorDetails[0]['designation'] == 'Unipole(Neon) Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 560000;
                    //     } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (728000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 728000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (873600 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 873600;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1164800 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 1164800;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2912000 / $numberDaysInYear);
                    //             $adsPrice = ($amountPerDay * $daysRemaining);
                    //                 $adsPrice = ($adsPrice * $getOutdoorDetails[0]['side']);
                    //                 $adsPrice = round($adsPrice, 2);
                    //                 $adsPricePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPricePYears = ($adsPricePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPricePYears = round($adsPricePYears, 2);
                    //                 $legalFee = (1 / 100) * $adsPrice;
                    //                 $legalFee = round($legalFee, 2);
                    //                 $legalFeePYears = (1 / 100) * $adsPricePYears;
                    //                 $legalFeePYears = round($legalFeePYears, 2);
                    //             $getRate = 2912000;
                    //     }
                    // }
                    
                    
                    // Get pricing from database
                    $pricing = OutdoorSignagePricing::where('designation', $getOutdoorDetails->designation)
                        ->where('street_category', $getOutdoorDetails->street_category)
                        ->where(function($query) use ($getOutdoorDetails) {
                            $query->whereNull('dimension_max')
                                ->orWhere('dimension_max', '>=', $getOutdoorDetails->dimension_sign);
                        })
                        ->where(function($query) use ($getOutdoorDetails) {
                            $query->whereNull('dimension_min')
                                ->orWhere('dimension_min', '<=', $getOutdoorDetails->dimension_sign);
                        })
                        ->first();
                
                    if (!$pricing) {
                        return response()->json([
                            'success' => false, 
                            'data' => 'Pricing not configured for this signage type: ' . $getOutdoorDetails->designation
                        ], $this->errorStatusCode);
                    }
                
                    // Calculate price based on dimension
                    $dimension = $getOutdoorDetails->dimension_sign;
                    $baseRate = $pricing->base_rate;
                            
                    // Apply dimension multiplier if needed
                    if ($pricing->dimension_max && $dimension > $pricing->dimension_max) {
                        $baseRate = $baseRate * ($dimension / $pricing->dimension_max) * $pricing->multiplier;
                    }
                
                    // Calculate daily or weekly rate
                    if ($pricing->calculation_type === 'daily') {
                        $amountPerDay = $baseRate / $numberDaysInYear;
                        $adsPrice = $amountPerDay * $daysRemaining;
                        $adsPricePYears = $amountPerDay * $daysRemainingPYears;
                    } else {
                        // Weekly calculation
                        $adsPrice = $baseRate * $weeksRemaining;
                        $adsPricePYears = $baseRate * $weeksRemainingPYears;
                    }
                
                    // Apply side multiplier
                    $adsPrice = $adsPrice * $getOutdoorDetails->side;
                    $adsPricePYears = $adsPricePYears * $getOutdoorDetails->side;
                            
                    // Round values
                    $adsPrice = round($adsPrice, 2);
                    $adsPricePYears = round($adsPricePYears, 2);
                            
                    // Calculate legal fees
                    $legalFee = (1 / 100) * $adsPrice;
                    $legalFee = round($legalFee, 2);
                    $legalFeePYears = (1 / 100) * $adsPricePYears;
                    $legalFeePYears = round($legalFeePYears, 2);
                            
                    $getRate = $baseRate;
                    
                    
                    
                    
                    
                    $getApprovedDate = Carbon::parse($upgradeAndDowngradeObj->created_at);
                    // $getApprovedDate = Carbon::createFromDate(2023, 12, 01);
                    $getPaymentStartDate = $getApprovedDate;
                    // var_dump($getPaymentStartDate);die();
                    // $getPaymentStartDate = $getApprovedDate;
    
                    $lastDayofYear = Carbon::createFromDate(Carbon::now()->year, 12, 31);
                    $lastDayofYearLastYear = Carbon::createFromDate(Carbon::now()->year - 1, 12, 31);
                    $daysRemaining = $getPaymentStartDate->diffInDays($lastDayofYear);
                    $weeksRemaining = $getPaymentStartDate->diffInWeeks($lastDayofYear);
                    if($getPaymentStartDate->year == Carbon::now()->year){
                    $daysRemainingPYears = 0;
                    $weeksRemainingPYears = 0;
                    }else{
                    $daysRemainingPYears = $getPaymentStartDate->diffInDays($lastDayofYearLastYear);
                    $weeksRemainingPYears = $getPaymentStartDate->diffInWeeks($lastDayofYearLastYear);
                    }
                    $currentYear = Carbon::now()->year;
                    $numberDaysInYear = Carbon::createFromDate($currentYear, 12, 31)->dayOfYear;
                    // Log::info("Number of days in a year: $numberDaysInYear");
                    // Log::info("Number of days remaining: $daysRemaining");
                    // Log::info("Last Day of the year: $lastDayofYear");
                    // Log::info("Updated app date: $getPaymentStartDate");
                    
                    $daysRemaining = $daysRemaining + 1;
                    if($weeksRemaining < 52){
                        $weeksRemaining = $weeksRemaining + 1;
                    }
                    
                    
                    
                    // if($upgradedowngradeadd == '16 Sheet') {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 6) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (22400 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 22400;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (44800 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 44800;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (56000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 56000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (140000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 140000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 6) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 6) * 22400;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 22400;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6 ) * 44800;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 44800;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 56000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 56000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign']/6) * 140000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 140000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == '48 Sheet') {
                    //     if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (84000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 84000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (196000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate =  196000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (280000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 280000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (700000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 700000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 84000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 84000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 196000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 196000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 280000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 280000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 18) * 700000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 700000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == '96 Sheet') {
                    //     if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (168000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 168000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (392000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 392000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1120000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1120000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2800000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 36) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 168000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 168000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 392000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 392000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 1120000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1120000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $getYearRate = ($getOutdoorDetails[0]['dimension_sign'] / 36) * 2800000;
                    //             $amountPerDay = ($getYearRate / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd== 'Rooftop/Static') {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 36) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 560000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 840000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1120000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1120000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2800000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (784000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 784000;
                    //          } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1134000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1134000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1414000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1414000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3094000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3094000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1008000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1008000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1288000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1288000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1568000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1568000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3248000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3248000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1232000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1232000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1512000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1512000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1792000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1792000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3472000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3472000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1456000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1456000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1736000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1736000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2016000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2016000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3696000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3696000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Rooftop/Ultrawave') {
                    //     if ($getOutdoorDetails[0]['dimension_sign'] <= 36) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 840000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1260000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1260000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (4200000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4200000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 73) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1134000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1134000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1554000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1554000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1974000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1974000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (4494000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4494000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 109) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1428000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1428000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1848000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1848000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2268000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2268000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (4788000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4788000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1722000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1722000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (2142000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2142000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2562000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2562000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (5082000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 5082000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 145) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (2016000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2016000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (2422000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2422000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2856000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2856000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (5376000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 5376000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Unipole') {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 54) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (980000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 980000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1176000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1176000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1568000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1568000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3920000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3920000;
                    //         }
                    //     }elseif($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (1400000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (2240000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2240000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (5600000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 5600000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] >= 108) {
                    //         if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1400000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 1680000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 2240000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2240000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108) * 5600000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 5600000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Front/Back Lit') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 560000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 560000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 560000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 840000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 840000;
                    //         }
                            
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1120000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1120000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1120000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1120000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (2800000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2800000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 2800000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2800000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == '48 Sheet Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (126000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 126000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 126000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 126000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (294000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 294000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 294000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 294000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (420000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 420000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 420000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 420000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1050000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1050000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1050000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1050000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Wall Panel Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (840000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 840000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 840000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 840000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1260000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1260000;
                    //         }else if ($getOutdoorDetails[0]['dimension_sign'] > 18) {
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1260000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1260000;
                    //         }
                    //     }  elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (4200000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4200000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4200000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == '96 Sheet Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (252000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 252000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 252000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 252000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (588000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 588000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 588000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 588000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (1680000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 1680000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1680000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 18) {
                    //             $amountPerDay = (4200000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4200000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 18){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 18 ) * 4200000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4200000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Unipole Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (1820000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1820000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 1820000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1820000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (2184000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2184000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2184000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2184000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (2912000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2912000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 2912000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2912000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         if ($getOutdoorDetails[0]['dimension_sign'] <= 108) {
                    //             $amountPerDay = (7280000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 7280000;
                    //         }else if($getOutdoorDetails[0]['dimension_sign'] > 108){
                    //             $amountPerDay = ((($getOutdoorDetails[0]['dimension_sign'] / 108 ) * 7280000) / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 7280000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Wall Drape') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (224 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (224 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (315 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (315 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (385 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (385 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (672 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (672 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Wrap Around' ) {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (175 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (175 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (245 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (245 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (315 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (315 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (546 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (546 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     }
                    // } elseif (str_contains($upgradedowngradeadd, 'LED')) {
                    //     if($getOutdoorDetails[0]['dimension_sign'] <= 40) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (2600000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2600000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'   ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (3250000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3250000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (3900000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 3900000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (7800000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 7800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 80) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (4875000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 4875000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District'  ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (6175000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 6175000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (7150000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 7150000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (14300000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 14300000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 100) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (6825000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 6825000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (8775000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 8775000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (10400000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 10400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (20800000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 20800000;
                    //         }
                    //     } elseif ($getOutdoorDetails[0]['dimension_sign'] > 100) {
                    //         if ($getOutdoorDetails[0]['street_category'] == 'Normal  N' || $getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (10400000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 10400000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Airport/Business District' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (13000000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 13000000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Highways N' ||
                    //             $getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (15600000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate =  15600000;
                    //         } elseif ($getOutdoorDetails[0]['street_category'] == 'Encroachment Zone N') {
                    //             $amountPerDay = (31200000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //             $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemove = round($adsPriceRemove, 2);
                    //             $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //             $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //             $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //             $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //             $legalFeeRemove = round($legalFeeRemove, 2);
                    //             $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //             $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 31200000;
                    //         }
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Overhead Gantries/Directional signs(A)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
                    //                 $amountPerDay = (280000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 280000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
                    //                 $amountPerDay = (760000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 760000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
                    //                 $amountPerDay = (1240000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1240000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
                    //                 $amountPerDay = (1720000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1720000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
                    //                 $amountPerDay = (2200000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 2200000;
                    //         }
                    //     }else{
                    //                 $amountPerDay = (700000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 700000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Overhead Gantries/Directional signs(B)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
                    //                 $amountPerDay = (390000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 390000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
                    //                 $amountPerDay = (1192500 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1192500;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
                    //                 $amountPerDay = (1995000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1995000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
                    //                 $amountPerDay = (2797500 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 2797500;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
                    //                 $amountPerDay = (3600000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 3600000;
                    //         }
                    //     }else{
                    //                 $amountPerDay = (2100000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 2100000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Overhead Gantries/Directional signs(C)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         if($getOutdoorDetails[0]['dimension_sign'] <= 30) {
                    //                 $amountPerDay = (500000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 500000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 60) {
                    //                 $amountPerDay = (1625000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1625000;
                                
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <= 90) {
                    //                 $amountPerDay = (2750000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 2750000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] <=120) {
                    //                 $amountPerDay = (3375000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 3375000;
                    //         } elseif ($getOutdoorDetails[0]['dimension_sign'] > 120) {
                    //                 $amountPerDay = (5000000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 5000000;
                    //         }
                    //     }else{
                    //                 $amountPerDay = (1400000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1400000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Overhead Gantries/Directional signs Ultrawave(A)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (420000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 420000;
                    //     }else{
                    //                 $amountPerDay = (1050000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1050000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Overhead Gantries/Directional signs Ultrawave(B)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (1260000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1260000;
                    //     }else{
                    //                 $amountPerDay = (3150000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 3150000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Overhead Gantries/Directional signs Ultrawave(C)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (7000000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 7000000;
                    //     }else{
                    //                 $amountPerDay = (17500000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 17500000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Iconic Structure (Billboard)') {
                    //     $amountPerDay = (14000000 / $numberDaysInYear);
                    //     $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //     $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //     $adsPriceRemove = round($adsPriceRemove, 2);
                    //     $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //     $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //     $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //     $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //     $legalFeeRemove = round($legalFeeRemove, 2);
                    //     $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //     $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //     $getRate = 14000000;
                    // } elseif ($upgradedowngradeadd == 'Tank Farm') {
                    //     $amountPerDay = (70000 / $numberDaysInYear);
                    //     $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //     $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //     $adsPriceRemove = round($adsPriceRemove, 2);
                    //     $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //     $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //     $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //     $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //     $legalFeeRemove = round($legalFeeRemove, 2);
                    //     $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //     $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //     $getRate = 70000;
                    // } elseif ($upgradedowngradeadd == 'Water Advertisement') {
                    //     $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //     $adsPriceRemove = (44800 * $weeksRemaining);
                    //     $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //     $adsPriceRemovePYears = (44800 * $weeksRemainingPYears);
                    //     $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //     $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //     $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //     $getRate = 0;
                    // } elseif ($upgradedowngradeadd == 'Water Tank') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (336 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (336 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     }else{
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (504 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (504 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Water Tank Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (420 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (420 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     }else{
                    //         $getDimension = $getOutdoorDetails[0]['dimension_sign'];
                    //         $adsPriceRemove = (630 * $getDimension * $weeksRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = (630 * $getDimension * $weeksRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemove = (1/100) * $adsPriceRemove;
                    //         $getRate = 0;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Bridge Panel (Pedestrian/vehicular)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (1120000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1120000;
                    //     }else{
                    //                 $amountPerDay = (2800000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 2800000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Bridge Panel Ultrawave (Pedestrian/vehicular)') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (1680000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1680000;
                    //     }else{
                    //                 $amountPerDay = (4200000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 4200000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Advert on Bridge built on private sector Initiative') {
                    //     if($getOutdoorDetails[0]['street_category'] != 'Local & International Airport N') {
                    //                 $amountPerDay = (560000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 560000;
                    //     }else{
                    //                 $amountPerDay = (1400000 / $numberDaysInYear);
                    //                 $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //                 $getRate = 1400000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Advert on Bridge built on private sector Initiative Ultrawave') {
                    //         $amountPerDay = (840000 / $numberDaysInYear);
                    //         $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //         $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemove = round($adsPriceRemove, 2);
                    //         $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //         $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //         $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //         $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //         $legalFeeRemove = round($legalFeeRemove, 2);
                    //         $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //         $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //         $getRate = 840000;
                    // } elseif ($upgradedowngradeadd == 'Unipole(Neon)') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
                    //             $amountPerDay = (350000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 350000;
                    //     } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 560000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (672000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 672000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (896000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 896000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2240000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2240000;
                    //     }
                    // } elseif ($upgradedowngradeadd == 'Unipole(Neon) Ultrawave') {
                    //     if($getOutdoorDetails[0]['street_category'] == 'Rural Zone N') {
                    //             $amountPerDay = (560000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 560000;
                    //     } elseif($getOutdoorDetails[0]['street_category'] == 'Normal Zone N') {
                    //             $amountPerDay = (728000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 728000;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Restrictive Zone N') {
                    //             $amountPerDay = (873600 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 873600;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'High Street N') {
                    //             $amountPerDay = (1164800 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 1164800;
                    //     } elseif ($getOutdoorDetails[0]['street_category'] == 'Local & International Airport N') {
                    //             $amountPerDay = (2912000 / $numberDaysInYear);
                    //             $adsPriceRemove = ($amountPerDay * $daysRemaining);
                    //                 $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemove = round($adsPriceRemove, 2);
                    //                 $adsPriceRemovePYears = ($amountPerDay * $daysRemainingPYears);
                    //                 $adsPriceRemovePYears = ($adsPriceRemovePYears * $getOutdoorDetails[0]['side']);
                    //                 $adsPriceRemovePYears = round($adsPriceRemovePYears, 2);
                    //                 $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    //                 $legalFeeRemove = round($legalFeeRemove, 2);
                    //                 $legalFeeRemovePYears = (1 / 100) * $adsPriceRemovePYears;
                    //                 $legalFeeRemovePYears = round($legalFeeRemovePYears, 2);
                    //             $getRate = 2912000;
                    //     }
                    // }
                    
                    
                    
                    // Get pricing from database
                    $pricing = OutdoorSignagePricing::where('designation', $upgradedowngradeadd)
                        ->where('street_category', $getOutdoorDetails->street_category)
                        ->where(function($query) use ($getOutdoorDetails) {
                            $query->whereNull('dimension_max')
                                ->orWhere('dimension_max', '>=', $getOutdoorDetails->dimension_sign);
                        })
                        ->where(function($query) use ($getOutdoorDetails) {
                            $query->whereNull('dimension_min')
                                ->orWhere('dimension_min', '<=', $getOutdoorDetails->dimension_sign);
                        })
                        ->first();
                
                    if (!$pricing) {
                        return response()->json([
                            'success' => false, 
                            'data' => 'Pricing not configured for this signage type: ' . $upgradedowngradeadd
                        ], $this->errorStatusCode);
                    }
                
                    // Calculate price based on dimension
                    $dimension = $getOutdoorDetails->dimension_sign;
                    $baseRate = $pricing->base_rate;
                            
                    // Apply dimension multiplier if needed
                    if ($pricing->dimension_max && $dimension > $pricing->dimension_max) {
                        $baseRate = $baseRate * ($dimension / $pricing->dimension_max) * $pricing->multiplier;
                    }
                
                    // Calculate daily or weekly rate
                    if ($pricing->calculation_type === 'daily') {
                        $amountPerDay = $baseRate / $numberDaysInYear;
                        $adsPriceRemove = $amountPerDay * $daysRemaining;
                        $adsPriceRemovePYears = $amountPerDay * $daysRemainingPYears;
                    } else {
                        // Weekly calculation
                        $adsPriceRemove = $baseRate * $weeksRemaining;
                        $adsPriceRemovePYears = $baseRate * $weeksRemainingPYears;
                    }
                
                    // Apply side multiplier
                    $adsPriceRemove = $adsPrice * $getOutdoorDetails->side;
                    $adsPriceRemovePYears = $adsPricePYears * $getOutdoorDetails->side;
                            
                    // Round values
                    $adsPriceRemove = round($adsPrice, 2);
                    $adsPriceRemovePYears = round($adsPricePYears, 2);
                            
                    // Calculate legal fees
                    $legalFeeRemove = (1 / 100) * $adsPrice;
                    $legalFeeRemove = round($legalFee, 2);
                    $legalFeeRemovePYears = (1 / 100) * $adsPricePYears;
                    $legalFeeRemovePYears = round($legalFeePYears, 2);
                            
                    $getRate = $baseRate;
                    
                    
                    
                    
                    if($billingInput['streetClassification'] == "Not Applicable"){
                    $legalFee = (1 / 100) * $billingInput['ads_price'];
                    $getRate = $billingInput['price_rate'];
                    $amountPerDay = ($getOutdoorDetails->price_rate / $numberDaysInYear);
                    $amountPerDayAdd = ($getRate / $numberDaysInYear);
                    $adsPrice = ($amountPerDay * $daysRemaining);
                    $adsPrice = ($adsPrice * $getOutdoorDetails->side);
                    $adsPrice = round($adsPrice, 2);
                    $adsPriceRemove = ($amountPerDayAdd * $daysRemaining);
                    $adsPriceRemove = ($adsPriceRemove * $getOutdoorDetails->side);
                    $adsPriceRemove = round($adsPriceRemove, 2);
                    $legalFeeRemove = (1 / 100) * $adsPriceRemove;
                    }
                    
                    $totalFee = bcadd($adsPrice, $legalFee, 2);
                    $getArrears = AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $outdoorObj->id)->latest()->first();
                    $getBalance = $getArrears->balance;
                    
                    $totalFeeAdd = bcadd($adsPriceRemove, $legalFeeRemove, 2);
                    // $oldBalance = bcadd($totalFee, $getBalance, 2);
                    $newBalance = round($getBalance+$totalFeeAdd-$totalFee, 2);
                    // $newBalance = bcadd($totalFee, $getBalance, 2);
                    ThirdPartyUpgradeAndDowngrade::where('id', $upgradeAndDowngradeObj->id)->update(['bill_status' => 'generated']);
                    ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id',  $outdoorObj->id)->update(['designation' => $upgradedowngradeadd,'ads_price' => $adsPriceRemove, 'legal_fee' => $legalFeeRemove, 'price_rate' => $getRate]);
                    AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $outdoorObj->id)->update(['ads_price' => $adsPriceRemove, 'legal_fee' => $legalFeeRemove,
                                            'price_rate' => $getRate, 'balance' => $newBalance]);
                                            
                    
                     
                            
                     $startdate = $getApprovedDate;
                     $enddate = Carbon::parse($upgradeAndDowngradeObj->created_at)->endOfYear();
                     
                     $userdetails = User::where('id', $outdoorObj->user_id)->first();
                    
                     $amountInWord = UtilMethods::getAmountInWords($initial_amount + $totalFeeAdd - $totalFee);
             
                     $commentt = strtoupper($upgradedowngradeadd);
                            
                     $data = [
                          'date' => Carbon::now()->format('jS F, Y'),
                          'company' => $userdetails->business_name,
                          'address' => $outdoorObj->address_proposed_site,
                          'email' => $userdetails->email,
                          'cust_id' => $outdoorObj->user_id,
                          'ref_no' => $outdoorObj->payment_id,
                          'user_type' => $userdetails->user_type,
                          'initial_amount_due' => $initial_amount,
                          'debit' => $totalFeeAdd,
                          'credit' => $totalFee,
                          'note_type' => 'change',
                          'amount_in_words' => $amountInWord,
                          'startDate' => $startdate->format('jS F, Y'),
                          'endDate' => $enddate->format('jS F, Y'),
                          'comment' => 'ADDITIONAL DEBIT NOTE FOR ' . $commentt . ' RATE'
                          
                     ];
                     
                    
                     $pdf = Pdf::loadView('CreditDebitNote', compact('data'))->setPaper('A3');
                     $fileObj = $pdf->download('CreditDebitNote.pdf');
                    
                      $userObj = User::find($outdoorObj->user_id);
                       $emailcontent = EmailContent::getContent('outdoor_debit_credit_content');
                     $body = $emailcontent;
                     $templateData = [
                         'body' => $body,
                         'name' => $userObj->business_name
                     ];
                     $templateContent = view('hodeoitemplate', $templateData)->render();
                     UtilMethods::debitnotethirdparty($userObj->email, $templateContent, $fileObj);
                     
                     $pdfContent = $pdf->output();
                     $filename = 'BillInvoice_' . now()->format('YmdHis') . '.pdf';
                     $relativePath = 'invoices/' . $filename;
                     $savePath = public_path($relativePath); // Full path to /public/invoices/
                
                     if (!File::exists(public_path('invoices'))) {
                         File::makeDirectory(public_path('invoices'), 0755, true);
                     }
                     file_put_contents($savePath, $pdfContent);
                     $fileUrl = config('app.url') . '/public/' . $relativePath;
                     $adnewrec = new ThirdPartyReceipts();
                     $adnewrec->link = $fileUrl;
                     $adnewrec->type = "Credit / Debit";
                     // $tosaveoutd = ThirdPartyOutdoor::find($getOutDoorObj[$i]['id']);
                     $outdoorObj->thirdpartyreceipts()->save($adnewrec);
    
                    return response()->json(['success' => true, 'adsPrice' => $adsPriceRemove, 'legalFee' => $legalFeeRemove, 'totalFee' => $newBalance], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'Bill has been generated'], $this->errorStatusCode);
            }
       } else {
           return response()->json(['success' => false, 'data' => 'your role can not assess the resource'], $this->errorStatusCode);
       }
   }

   public function hodBillingApproval($outdoorId, $requestId) {
        if(Auth::user()->department == 'Billings' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);

            if($upgradeandDowngradeObj->hod_billngs == null) {
                ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['hod_billngs' => 'approved']);
                $getResponse = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'the bill has already been verified'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you are not in the department or role to access resource'], $this->errorStatusCode);
        }
   }

   public function hodBillingDisapproval($outdoorId, $requestId) {
       if(Auth::user()->department == 'Billings' && Auth::user()->admin_role_id == 5) {
           $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
           $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);

           if($upgradeandDowngradeObj->hod_billngs == null) {
               ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['hod_billngs' => 'disapproved']);
               $getResponse = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
               return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
           } else {
               return response()->json(['success' => false, 'data' => 'the bill has already been verified'], $this->errorStatusCode);
           }
       } else {
           return response()->json(['success' => false, 'data' => 'you are not in the department or role to access resource'], $this->errorStatusCode);
       }
   }

   public function hodRegistryApproval($outdoorId, $requestId) {
       if(Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
           $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
           $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);

           if($upgradeandDowngradeObj->hod_registry == null) {
               ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['hod_registry' => 'approved']);
               $getResponse = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
               return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
           } else {
               return response()->json(['success' => false, 'data' => 'the bill has already been verified'], $this->errorStatusCode);
           }
       } else {
           return response()->json(['success' => false, 'data' => 'you are not in the department or role to access resource'], $this->errorStatusCode);
       }
   }

    public function hodRegistryDisapproval($outdoorId, $requestId) {
        if(Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $upgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);

            if($upgradeandDowngradeObj->hod_registry == null) {
                ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->update(['hod_registry' => 'disapproved']);
                $getResponse = ThirdPartyUpgradeAndDowngrade::where('id', $upgradeandDowngradeObj->id)->get();
                return response()->json(['success' => true, 'data' => $getResponse], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'the bill has already been verified'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you are not in the department or role to access resource'], $this->errorStatusCode);
        }
    }

    /**
     * @throws \NumberToWords\Exception\NumberToWordsException
     * @throws \NumberToWords\Exception\InvalidArgumentException
     */
    public function sendBillAfterUpgradeorDowngrade($outdoorId, $requestId) {

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $getUpgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);

        if(Auth::user()->id == (string)$getOutdoorObj->assigned_officer) {
            if ($getUpgradeandDowngradeObj->bill_status == 'generated') {
                if ($getUpgradeandDowngradeObj->hod_billngs == 'approved' && $getUpgradeandDowngradeObj->hod_registry == 'approved') {

                    $getThirdPartyOutdoor = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)
                        ->with(['user', 'thirdPartyOutdoorSignDetails', 'appHistoryThirdPartyOutdoor'])->get();
                    $userDetails = KycThirdParty::where('user_id', $getThirdPartyOutdoor[0]['user_id'])->with(['user'])->get();
                    $getThirdPartyDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)->get();
                    $getOutstandingBalance = AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOutdoorObj->id)->get()->last();

                    $adsPrice = $getThirdPartyDetails[0]['ads_price'];
                    $legalFee = $getThirdPartyDetails[0]['legal_fee'];
                    $totalFee = bcadd($adsPrice, $legalFee, 2);

                    $amountInWord = UtilMethods::getAmountInWords($totalFee);

                    $getDate = Carbon::parse($getThirdPartyDetails[0]['updated_at']);
                    $getYear = $getDate->year;
                    $data = [
                        'custId' => $userDetails[0]['id'],
                        'email' => $userDetails[0]['email'],
                        'userType' => $userDetails[0]['user_type'],
                        'company' => $userDetails[0]['company_name'],
                        'officeAddress' => $userDetails[0]['office_address'],
                        'billYear' => $getYear,
                        'signageLocation' => $getThirdPartyDetails[0]['address_proposed_site'],
                        'appId' => $getThirdPartyDetails[0]['id'],
                        'billStartDate' => Carbon::parse($getThirdPartyDetails[0]['updated_at'])->format('m-d-y'),
                        'billEndDate' => Carbon::now()->endOfYear()->format('m-d-y'),
                        'dimension' => $getThirdPartyOutdoor[0]['thirdPartyOutdoorSignDetails']['dimension_sign'],
                        'signage' => $getThirdPartyOutdoor[0]['thirdPartyOutdoorSignDetails']['designation'],
                        'zone' => $getThirdPartyOutdoor[0]['thirdPartyOutdoorSignDetails']['street_category'],
                        'side' => $getThirdPartyOutdoor[0]['thirdPartyOutdoorSignDetails']['side'],
                        'rate' => $getOutstandingBalance['price_rate'],
                        'adsFee' => $getOutstandingBalance['ads_price'],
                        'legalFee' => $getOutstandingBalance['legal_fee'],
                        'totalFee' => $totalFee,
                        'amountInWord' => $amountInWord
                    ];

                    $pdf = Pdf::loadView('BillInvoice', compact('data'))->setPaper('A3');
                    $fileObj = $pdf->download('BillInvoice.pdf');

                    $getUserObj = User::find($userDetails[0]['user']['id']);
                    $emailcontent = EmailContent::getContent('outdoor_bill_content');

                    $body = $emailcontent;
                    $templateData = [
                        'body' => $body,
                        'name' => $getUserObj->company_name
                    ];
                    $templateContent = view('hodeoitemplate', $templateData)->render();
                    UtilMethods::outdoorSignageInvoiceBill($getUserObj->email, $templateContent, $fileObj);
                    
                    $pdfContent = $pdf->output();
                    $filename = 'BillInvoice_' . now()->format('YmdHis') . '.pdf';
                    $relativePath = 'invoices/' . $filename;
                    $savePath = public_path($relativePath); // Full path to /public/invoices/
                    
                    if (!File::exists(public_path('invoices'))) {
                        File::makeDirectory(public_path('invoices'), 0755, true);
                    }
                    file_put_contents($savePath, $pdfContent);
                    $fileUrl = config('app.url') . '/public/' . $relativePath;
                    $adnewrec = new ThirdPartyReceipts();
                    $adnewrec->link = $fileUrl;
                    $adnewrec->type = "Invoice";
                    $tosaveoutd = ThirdPartyOutdoor::find($getOutdoorObj->id);
                    $tosaveoutd->thirdpartyreceipts()->save($adnewrec);
                    
                    // return response()->json(['success' => true, 'data' => 'Invoice sent', 'signage bill' => $adsPrice, 'legal fee' => $legalFee],
                    //     $this->successStatusCode);
                    return true;
                } else {
                    return response()->json(['success' => false, 'data' => 'Either of the HOD has approved the bills'], $this->errorStatusCode);
                }
            } else {
                return response()->json(['success' => false, 'data' => 'Billing has not been generated'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you are not the rm to the application'], $this->errorStatusCode);
        }
    }

    public function sendUpgradeorDowngradeApprovalLetter($outdoorId, $requestId) {

        $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
        $getUpgradeandDowngradeObj = ThirdPartyUpgradeAndDowngrade::find($requestId);

        // if(Auth::user()->id == (string)$getOutdoorObj->assigned_officer) {
            if ($getUpgradeandDowngradeObj->bill_payment_status == 'paid') {
                ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->update(['active_status' => null, 'ongoing_request' => null]);
                $userDetails = KycThirdParty::where('user_id', $getOutdoorObj->user_id)->with(['user'])->get();
                $getThirdPartyDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)->get();
                $getUpgradeandDowngrade = ThirdPartyUpgradeAndDowngrade::where('third_party_outdoor_id', $getOutdoorObj->id)->get();

                $outdoorSignageName = ucwords($getUpgradeandDowngrade[0]['request_outdoor_signage']);
                $outdoorRequest = $getUpgradeandDowngrade[0]['type_charges'];
                $outdoorWidth = $getThirdPartyDetails[0]['breadth_sign'];
                $outdoorLength = $getThirdPartyDetails[0]['length_sign'];
                $outdoorDimension = $getThirdPartyDetails[0]['dimension_sign'];
                $outdoorSide = $getThirdPartyDetails[0]['number_faces'];
                $signageSite = $getOutdoorObj->address_proposed_site;
                $date = Carbon::parse($getUpgradeandDowngrade[0]['updated_at'])->format('jS F, Y');
                $refno = $getOutdoorObj->application_id;
                if ($getOutdoorObj->application_id == null) {
                    $refno = UtilMethods::outdoorRequestAppId();
                }

                $data = [
                    'companyName' => $userDetails[0]['company_name'],
                    'officeAddress' => $userDetails[0]['office_address'],
                    'signageId' => $getOutdoorObj->id,
                    'signageName' => $outdoorSignageName,
                    'signageWidth' => $outdoorWidth,
                    'signageHeight' => $outdoorLength,
                    'signageDimension' => $outdoorDimension,
                    'signageSide' => $outdoorSide,
                    'signageDuration' => 'Annual',
                    'signageRequest' => $outdoorRequest,
                    'signageSite' => $signageSite,
                    'appId' => $refno,
                    'date' => $date
                ];

                $pdf = Pdf::loadView('upgradeanddowngrade', compact('data'));
                $fileObj = $pdf->download('upgradeanddowngrade.pdf');

                $getUserObj = User::find($userDetails[0]['user']['id']);
                $emailcontent = EmailContent::getContent('outdoor_upgrade_downgrade_content');

                $body = $emailcontent . $outdoorRequest;
                $templateData = [
                    'body' => $body,
                    'name' => $getUserObj->company_name
                ];
                $templateContent = view('hodeoitemplate', $templateData)->render();
                UtilMethods::outdoorSignageUpgradeandDowngrade($getUserObj->email, $templateContent, $fileObj);
                
                $pdfContent = $pdf->output();
                $filename = 'ApprovalLetter_' . now()->format('YmdHis') . '.pdf';
                $relativePath = 'invoices/' . $filename;
                $savePath = public_path($relativePath); // Full path to /public/invoices/
                
                if (!File::exists(public_path('invoices'))) {
                    File::makeDirectory(public_path('invoices'), 0755, true);
                }
                file_put_contents($savePath, $pdfContent);
                $fileUrl = config('app.url') . '/public/' . $relativePath;
                $adnewrec = new ThirdPartyReceipts();
                $adnewrec->link = $fileUrl;
                $adnewrec->type = "Letter";
                // $tosaveoutd = ThirdPartyOutdoor::find($getOutDoorObj[$i]['id']);
                $getOutdoorObj->thirdpartyreceipts()->save($adnewrec);
                
                return response()->json(['success' => true, 'data' => 'Approval letter sent'], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'Payment has not been made on the upgrade/downgrade signage'], $this->errorStatusCode);
            }
        // } else {
        //     return response()->json(['success' => false, 'data' => 'Only application RM can send the letter'], $this->errorStatusCode);
        // }
    }

   public static function capitalizeDesignation($designation) {
        return ucwords(strtolower($designation));
    }

    ///////////////////////////////////////////////ADMIN OUTDOOR SIGNAGE TRANSFER////////////////////////////////////////////////////

//    public function setSiteInspectionStatusforTransfer($outdoorId, $requestId) {
//        if (Auth::user()->department == 'Registry' && (Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3)) {
//            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
//            if($outdoorObj->assigned_officer == (string) Auth::user()->id) {
//                $upgradeDowngradeObj  =  ThirdPartyOutdoorTransfer::find($requestId);
//                if($upgradeDowngradeObj->site_inspection_status == 'pending') {
//                    ThirdPartyOutdoorTransfer::where('id', $upgradeDowngradeObj->id)->update(['site_inspection_status' => 'done',
//                                            'hod_recommendation' => 'pending']);
//                    $response = ThirdPartyOutdoorTransfer::where('id', $upgradeDowngradeObj->id)->get();
//                    return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
//                } else {
//                    return response()->json(['success' => false, 'data' => 'site inspection status is not pending'], $this->errorStatusCode);
//                }
//            } else {
//                return response()->json(['success' => false, 'data' => 'you are not the assigned rm to this application'], $this->errorStatusCode);
//            }
//        } else {
//            return response()->json(['success' => false, 'data' => 'Either you are not an rm or do not belong to regsitry department'], $this->errorStatusCode);
//        }
//    }

    public function recommendforSiteTransferApproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                

        if (Auth::user()->department == 'Registry' && (Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3)) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $outdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);
            if($outdoorTransferObj->rm_recommendation == 'pending') {
                ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->update(['rm_recommendation' => 'approved',
                    'hod_recommendation' => 'pending', 'rm_comment' => $request['comment']]);
                $response = ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'RM has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an Hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }

    public function recommendforSiteTransferDisapproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                

        if (Auth::user()->department == 'Registry' && (Auth::user()->admin_role_id == 2 || Auth::user()->admin_role_id == 3)) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $outdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);
            if($outdoorTransferObj->rm_recommendation == 'waiting') {
                ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->update(['rm_recommendation' => 'disapproved',
                    'hod_recommendation' => 'pending', 'rm_comment' => $request['comment']]);
                $response = ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'RM has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an Hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }
    
    public function hodrecommendforSiteTransferApproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                

        if (Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $outdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);
            if($outdoorTransferObj->hod_recommendation == 'pending') {
                ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->update(['hod_recommendation' => 'approved',
                    'md_approval' => 'pending', 'hod_comment' => $request['comment']]);
                $response = ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'RM has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an Hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }

    public function hodrecommendforSiteTransferDisapproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                

        if (Auth::user()->department == 'Registry' && Auth::user()->admin_role_id == 5) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $outdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);
            if($outdoorTransferObj->hod_recommendation == 'pending') {
                ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->update(['hod_recommendation' => 'disapproved',
                    'md_approval' => 'pending', 'hod_comment' => $request['comment']]);
                $response = ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->get();
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'RM has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'Either you are not an Hod or do not belong to registry department'], $this->errorStatusCode);
        }
    }

    public function mdSiteTransferApproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                
        if (Auth::user()->admin_role_id == 6) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $outdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);
            if($outdoorTransferObj->md_approval == 'pending') {
                ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->update(['md_approval' => 'approved', 'md_comment' => $request['comment']]);
                // ThirdPartyOutdoor::where('id', $outdoorObj->id)->update(['active_status' => null]);
                ThirdPartyOutdoor::where('id', $outdoorObj->id)->update(['user_id' => $outdoorTransferObj->new_owner_id,
                        'ongoing_request' => null]);
                $response = ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->get();
                
                
                
                    $getResponse = ThirdPartyOutdoor::where('id', $outdoorId)->get();
                    $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_completed_content');
                    $subject = $emailsubject;
                    $body = $emailcontent;
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                    
                    
                    // $getResponse = ThirdPartyOutdoor::where('id', $outdoorId)->get();
                    $getApplicant = User::where('id', $outdoorObj->user_id)->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_removed_content');
                    $subject = $emailsubject;
                    $body = $emailcontent;
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                
                
                
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'MD has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you need md priviledge to take this action'], $this->errorStatusCode);
        }
    }

    public function mdSiteTransferDisapproval($outdoorId, $requestId, Request $request) {
        $validate = Validator::make($request->all(), [
                    'comment' => 'nullable|string'
                ]);

                if($validate->fails()) {
                    return response()->json(['errors' => $validate->errors()->all()], $this->errorStatusCode);
                }
                
        if (Auth::user()->admin_role_id == 6) {
            $outdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $outdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);
            if($outdoorTransferObj->md_approval == 'pending') {
                ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->update(['md_approval' => 'disapproved', 'md_comment' => $request['comment']]);
                $response = ThirdPartyOutdoorTransfer::where('id', $outdoorTransferObj->id)->get();
                $getResponse = ThirdPartyOutdoor::where('id', $outdoorId)->get();
                    $getApplicant = User::where('id', $getResponse[0]['user_id'])->get();
                    $getApplicantEmail = $getApplicant[0]['email'];
                    $emailsubject = EmailContent::getContent('outdoor_decline_approve_subject');
                    $emailcontent = EmailContent::getContent('outdoor_decline_content');
                    $subject = $emailsubject;
                    $body = $emailcontent. " ". $request['comments'] . " ";
            
                    $templateData = [
                        'body' => $body,
                        'name' => $getApplicant[0]['business_name'],
            
                    ];
                    $templateContent = view('emailtemplate', $templateData)->render();
                    $this->composeEmailV2($subject, $getApplicantEmail,  $templateContent);
                return response()->json(['success' => true, 'data' => $response], $this->successStatusCode);
            } else {
                return response()->json(['success' => false, 'data' => 'MD has already recommend this application'], $this->errorStatusCode);
            }
        } else {
            return response()->json(['success' => false, 'data' => 'you need md priviledge to take this action'], $this->errorStatusCode);
        }
    }

    // things to work on include ending of the year to add annual bills to the balance on ground.

    public function generateTransferBill($outdoorId, $requestId) {
        if((Auth::user()->department == 'Billings') && (Auth::user()->admin_role_id  == 2 || Auth::user()->admin_role_id == 3)) {
            $getOutdoorObj = ThirdPartyOutdoor::find($outdoorId);
            $getOutdoorTransferObj = ThirdPartyOutdoorTransfer::find($requestId);

            if($getOutdoorTransferObj->md_approval == 'approved') {
                if($getOutdoorTransferObj->transfer_outstanding == false) {
                    $getOutstandingBalance = AppHistoryThirdPartyOutdoor::where('third_party_outdoor_id', $getOutdoorObj->id)->get()->last();
                    $getThirdPartyDetails = ThirdPartyOutdoorSignDetails::where('third_party_outdoor_id', $getOutdoorObj->id)->get();
                    $getThirdPartyOutdoor = ThirdPartyOutdoor::where('id', $getOutdoorObj->id)->get();
                    $userDetails = User::where('id', $getOutdoorObj->us